import "jsr:@supabase/functions-js/edge-runtime.d.ts";
import { createClient } from "jsr:@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, content-type",
  "Access-Control-Allow-Methods": "GET, POST, OPTIONS",
};

// 删除指定目录及其下的文件
async function deleteDirectory(supabase, bucket: string, directory: string) {
  try {
    const { data, error } = await supabase.storage.from(bucket).list(directory, {
      limit: 1000,
    });

    if (error) {
      console.error(`Failed to list files in directory "${directory}":`, error.message);
      return;
    }

    if (data && data.length > 0) {
      const pathsToDelete = data.map((file) => `${directory}/${file.name}`);
      const { error: deleteError } = await supabase.storage.from(bucket).remove(pathsToDelete);

      if (deleteError) {
        console.error(`Failed to delete files in directory "${directory}":`, deleteError.message);
      } else {
        console.log(`Successfully deleted directory "${directory}" and its contents.`);
      }
    }
  } catch (err) {
    console.error(`Error deleting directory "${directory}":`, err);
  }
}

// 计算特定周期的保留时间戳
function calculateRetentionTimestamp(cycle: 'daily' | 'hourly'): {
  retentionTimestamp: number,
  currentPeriodTimestamp: number
} {
  const now = new Date()

  let currentPeriodTimestamp: number
  let retentionTimestamp: number

  if (cycle === 'daily') {
    // 当前周期的时间戳(每天凌晨3点)
    const currentPeriod = new Date(now)
    currentPeriod.setHours(3, 0, 0, 0)
    currentPeriodTimestamp = currentPeriod.getTime() / 1000

    // 保留最近7天的数据，即7个凌晨3点周期
    const retentionPeriod = new Date(now)
    retentionPeriod.setDate(retentionPeriod.getDate() - 7)
    retentionPeriod.setHours(3, 0, 0, 0)
    retentionTimestamp = retentionPeriod.getTime() / 1000
  } else {
    // 当前周期的时间戳(每小时整点)
    const currentPeriod = new Date(now)
    currentPeriod.setMinutes(0, 0, 0)
    currentPeriodTimestamp = currentPeriod.getTime() / 1000

    // 保留最近7个小时的数据
    const retentionPeriod = new Date(now)
    retentionPeriod.setHours(retentionPeriod.getHours() - 7, 0, 0, 0)
    retentionTimestamp = retentionPeriod.getTime() / 1000
  }

  return { retentionTimestamp, currentPeriodTimestamp }
}

// 清理存储桶中过期数据
async function cleanupStorage(supabase: any, bucketName: string, category: 'app-category' | 'user-rank') {
  const cycle = category === 'app-category' ? 'daily' : 'hourly'
  const { retentionTimestamp, currentPeriodTimestamp } = calculateRetentionTimestamp(cycle)
  const deletedDirectories: string[] = []

  try {
    // 获取指定目录下的所有子目录
    const { data, error } = await supabase.storage.from(bucketName).list(category + '/', {
      limit: 100,
      offset: 0
    })

    if (error) throw error

    // 过滤并删除过期目录
    for (const item of data) {
      const dirTimestamp = parseInt(item.name)

      // 确保不删除当前周期的目录，且删除早于保留时间的目录
      if (dirTimestamp < retentionTimestamp && dirTimestamp !== currentPeriodTimestamp) {
        await deleteDirectory(supabase, bucketName, `${category}/${item.name}`)
        deletedDirectories.push(item.name)
        console.log(`删除过期目录: ${category}/${item.name}`)
      }
    }

    return {
      category,
      cycle,
      totalDeleted: deletedDirectories.length,
      deletedDirectories,
      currentPeriodTimestamp: currentPeriodTimestamp.toString()
    }
  } catch (err) {
    console.error(`清理 ${category} 存储时发生错误:`, err)
    return {
      category,
      cycle,
      totalDeleted: 0,
      error: err.message
    }
  }
}



// 主逻辑处理
Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response("ok", { headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get("SUPABASE_URL");
    const supabaseAnonKey = Deno.env.get("SUPABASE_ANON_KEY");
    const authHeader = req.headers.get("Authorization");

    if (!supabaseUrl || !supabaseAnonKey || !authHeader) {
      throw new Error("Missing Supabase configuration or Authorization header.");
    }

    const supabase = createClient(supabaseUrl, supabaseAnonKey, {
      global: { headers: { Authorization: authHeader } },
    });

    console.log("Starting cleanup process...");

    // 清理存储并收集结果
    const appCategoryResult = await cleanupStorage(supabase, 'cache', 'app-category')
    const userRankResult = await cleanupStorage(supabase, 'cache', 'user-rank')

    return new Response(
      JSON.stringify({
        message: '存储清理完成',
        results: [appCategoryResult, userRankResult]
      }),
      {
        headers: { 'Content-Type': 'application/json' }
      }
    )
  } catch (err) {
    console.error("Error in cleanup process:", err);
    return new Response(
      JSON.stringify({ code: 500, message: err.message, data: null }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" }, status: 500 }
    );
  }
});