// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

// Setup type definitions for built-in Supabase Runtime APIs
import "jsr:@supabase/functions-js/edge-runtime.d.ts"
import { createClient } from 'jsr:@supabase/supabase-js@2'

// 分页查询函数
async function fetchAllData(supabase, table: string, pageSize: number = 1000) {
  let allData: any[] = [];
  let offset = 0;

  while (true) {
    const { data, error } = await supabase
      .from(table)
      .select('*') // 可以根据需要调整字段
      .eq('is_show', true)
      .order('created_at', { ascending: true }) // 按照创建时间排序
      .range(offset, offset + pageSize - 1); // 分页范围

    if (error) {
      console.error(`Error fetching data from ${table}:`, error);
      throw error;
    }

    if (data && data.length > 0) {
      allData = allData.concat(data); // 合并当前页数据
      offset += pageSize; // 移动到下一页
    } else {
      break; // 如果没有更多数据，则退出循环
    }
  }

  return allData;
}

const getTimestampForMidnight3AM = () => {
  const timestamp = Math.floor(Date.now() / 3600000) * 3600; // 计算整点的时间戳
  return timestamp

};

Deno.serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders })
  }
  try {
    // const { name } = await req.json()
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    )

    // 获取所有 app 数据
    const allApps = await fetchAllData(supabase, 'app');
    // 获取当前时间的整点时间戳
    const timestamp = getTimestampForMidnight3AM();

    // 上传所有 app 数据到 storage
    const directory = `app-category/${timestamp}`;
    const bucketName = 'cache'; // 替换为存储桶名称
    // 分页参数
    const pageSize = 30;
    const totalCount = allApps.length;
    const paginatedApps = [];
    for (let i = 0; i < totalCount; i += pageSize) {
      paginatedApps.push(allApps.slice(i, i + pageSize));
    }

    // 遍历分页后的数据并上传
    for (let pageNum = 1; pageNum <= paginatedApps.length; pageNum++) {
      const pageApps = paginatedApps[pageNum - 1];
      const pageJson = JSON.stringify({
        total_count: totalCount,
        apps: pageApps,
      });

      const pageFileName = `${directory}/app_all_${pageNum}.json`;

      const { error: pageUploadError } = await supabase.storage
        .from(bucketName)
        .upload(pageFileName, new Blob([pageJson]), {
          contentType: 'application/json',
          upsert: true,
        });

      if (pageUploadError) {
        console.error(`Error uploading page ${pageNum} JSON:`, pageUploadError);
        return new Response(
          JSON.stringify({ error: `Failed to upload app_all_${pageNum}.json` }),
          { status: 500 }
        );
      }
    }

    // 按 category_id 分组
    const groupedData: Record<string, any[]> = {};
    allApps.forEach((app) => {
      if (!groupedData[app.category_id]) {
        groupedData[app.category_id] = [];
      }
      groupedData[app.category_id].push(app);
    });

    // 上传每个 category_id 的数据
    // 上传每个 category_id 的分页数据
    for (const [categoryId, apps] of Object.entries(groupedData)) {
      const totalCount = apps.length; // 当前分类的总数
      const pageSize = 30;
      const paginatedApps = [];
      for (let i = 0; i < totalCount; i += pageSize) {
        paginatedApps.push(apps.slice(i, i + pageSize));
      }

      // 遍历分页数据并上传
      for (let pageNum = 1; pageNum <= paginatedApps.length; pageNum++) {
        const pageApps = paginatedApps[pageNum - 1];
        const pageJson = JSON.stringify({
          total_count: totalCount,
          apps: pageApps,
        });

        // 文件名格式：{categoryId}_{pageNum}.json
        const categoryPageFileName = `${directory}/${categoryId}_${pageNum}.json`;

        const { error: categoryPageUploadError } = await supabase.storage
          .from(bucketName)
          .upload(categoryPageFileName, new Blob([pageJson]), {
            contentType: 'application/json',
            upsert: true,
          });

        if (categoryPageUploadError) {
          console.error(
            `Error uploading category ${categoryId} page ${pageNum} JSON:`,
            categoryPageUploadError
          );
          return new Response(
            JSON.stringify({
              error: `Failed to upload category ${categoryId} page ${pageNum} JSON`,
            }),
            { status: 500 }
          );
        }
      }
    }
    return new Response(
      JSON.stringify({
        message: 'All data uploaded successfully',
        fileName: allAppsFileName,
      }),
      {
        headers: { 'Content-Type': 'application/json' },
        status: 200
      }
    );
  } catch (err) {
    console.error('Unexpected error:', err);
    return new Response(JSON.stringify({ error: 'Internal server error' }), {
      headers: { 'Content-Type': 'application/json' },
      status: 500,
    });
  }
})




/* To invoke locally:

  1. Run `supabase start` (see: https://supabase.com/docs/reference/cli/supabase-start)
  2. Make an HTTP request:

  curl -i --location --request POST 'http://127.0.0.1:54321/functions/v1/hello-world' \
    --header 'Authorization: Bearer ' \
    --header 'Content-Type: application/json' \
    --data '{"name":"Functions"}'

*/
