package main

import (
	"context"
	"log"
	"os"
	"os/signal"
	"syscall"
	"time"

	"chain-sql/internal/api"
	"chain-sql/internal/blockchain"
	"chain-sql/internal/config"
	"chain-sql/internal/core"
	"chain-sql/internal/database"
	"chain-sql/internal/logger"
	"chain-sql/internal/metrics"

	"go.uber.org/zap"
)

func main() {
	// 1. 加载配置
	cfg, err := config.Load("configs/config.yaml")
	if err != nil {
		log.Fatalf("Failed to load config: %v", err)
	}

	// 2. 初始化日志
	if err := logger.Init(cfg.Log.Level, cfg.Log.Development); err != nil {
		log.Fatalf("Failed to initialize logger: %v", err)
	}
	defer logger.Sync()

	logger.Log.Info("ChainSQL starting",
		zap.Int64("chain_id", cfg.Chain.ChainID),
		zap.String("rpc_url", cfg.Chain.RPCURL),
		zap.String("log_level", cfg.Log.Level),
	)

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// 3. 连接数据库
	db, err := database.New(ctx, cfg.Database.DSN)
	if err != nil {
		logger.Log.Fatal("Failed to connect to database", zap.Error(err))
	}
	defer db.Close()
	logger.Log.Info("Database connected and schema initialized")

	// 4. 连接区块链节点
	ethClient, err := blockchain.NewClient(cfg.Chain.RPCURL)
	if err != nil {
		logger.Log.Fatal("Failed to connect to blockchain", zap.Error(err))
	}
	defer ethClient.Close()

	// 测试节点连接
	blockNum, err := ethClient.BlockNumber(ctx)
	if err != nil {
		logger.Log.Warn("Failed to get block number (is node running?)", zap.Error(err))
	} else {
		logger.Log.Info("Connected to Ethereum node", zap.Uint64("current_block", blockNum))
	}

	// 5. 启动 HTTP 服务器 (metrics, health checks)
	httpServer := api.NewServer(cfg.HTTP.Port, db)
	go func() {
		if err := httpServer.Start(); err != nil {
			logger.Log.Error("HTTP server error", zap.Error(err))
		}
	}()
	defer func() {
		shutdownCtx, shutdownCancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer shutdownCancel()
		if err := httpServer.Shutdown(shutdownCtx); err != nil {
			logger.Log.Error("HTTP server shutdown error", zap.Error(err))
		}
	}()

	// 6. 启动 Factory Listener
	factoryListener := core.NewFactoryListener(cfg, ethClient, db)
	go factoryListener.Start(ctx)

	// 7. 初始化 Contract Manager
	mgr := core.NewContractManager(db)
	go mgr.StartAutoReload(ctx, 30*time.Second) // 每30秒刷新一次列表

	// 8. 启动 Data Listener
	dataListener := core.NewDataListener(cfg, ethClient, db, mgr)
	go dataListener.Start(ctx)

	// 9. 初始化监控指标
	metrics.ActiveContracts.Set(0) // 初始值

	logger.Log.Info("All services started successfully",
		zap.Int("http_port", cfg.HTTP.Port),
	)

	// 10. 等待信号退出
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)
	<-sigChan

	logger.Log.Info("Shutting down gracefully...")
	cancel()
	time.Sleep(2 * time.Second) // 给各个 goroutine 一些时间清理

	logger.Log.Info("ChainSQL stopped")
}
