package blockchain

import (
	"context"
	"fmt"
	"math/big"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
)

// Client 定义了我们需要用到的链上操作接口，方便测试时 Mock
type Client interface {
	BlockNumber(ctx context.Context) (uint64, error)
	FilterLogs(ctx context.Context, query ethereum.FilterQuery) ([]types.Log, error)
	ChainID(ctx context.Context) (*big.Int, error)
	Close()
}

type EthClient struct {
	client *ethclient.Client
}

func NewClient(rawurl string) (*EthClient, error) {
	client, err := ethclient.Dial(rawurl)
	if err != nil {
		return nil, fmt.Errorf("failed to connect to eth node: %w", err)
	}
	return &EthClient{client: client}, nil
}

func (c *EthClient) Close() {
	c.client.Close()
}

func (c *EthClient) BlockNumber(ctx context.Context) (uint64, error) {
	return c.client.BlockNumber(ctx)
}

func (c *EthClient) FilterLogs(ctx context.Context, query ethereum.FilterQuery) ([]types.Log, error) {
	return c.client.FilterLogs(ctx, query)
}

func (c *EthClient) ChainID(ctx context.Context) (*big.Int, error) {
	return c.client.ChainID(ctx)
}
