package metrics

import (
	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promauto"
)

var (
	// EventsProcessed 事件处理计数器
	EventsProcessed = promauto.NewCounterVec(
		prometheus.CounterOpts{
			Name: "chainsql_events_processed_total",
			Help: "Total number of blockchain events processed",
		},
		[]string{"event_type", "status"}, // status: success, error
	)

	// EventProcessingDuration 事件处理延迟
	EventProcessingDuration = promauto.NewHistogramVec(
		prometheus.HistogramOpts{
			Name:    "chainsql_event_processing_duration_seconds",
			Help:    "Duration of event processing in seconds",
			Buckets: prometheus.DefBuckets,
		},
		[]string{"event_type"},
	)

	// SyncBlockHeight 同步区块高度
	SyncBlockHeight = promauto.NewGaugeVec(
		prometheus.GaugeOpts{
			Name: "chainsql_sync_block_height",
			Help: "Current synced block height",
		},
		[]string{"listener_type"}, // factory, data
	)

	// ActiveContracts 活跃合约数量
	ActiveContracts = promauto.NewGauge(
		prometheus.GaugeOpts{
			Name: "chainsql_active_contracts",
			Help: "Number of active contracts being monitored",
		},
	)

	// DatabaseOperations 数据库操作计数器
	DatabaseOperations = promauto.NewCounterVec(
		prometheus.CounterOpts{
			Name: "chainsql_database_operations_total",
			Help: "Total number of database operations",
		},
		[]string{"operation", "status"}, // operation: insert, update, delete, create_table
	)

	// DatabaseOperationDuration 数据库操作延迟
	DatabaseOperationDuration = promauto.NewHistogramVec(
		prometheus.HistogramOpts{
			Name:    "chainsql_database_operation_duration_seconds",
			Help:    "Duration of database operations in seconds",
			Buckets: prometheus.DefBuckets,
		},
		[]string{"operation"},
	)

	// BlockchainRPCCalls RPC 调用计数器
	BlockchainRPCCalls = promauto.NewCounterVec(
		prometheus.CounterOpts{
			Name: "chainsql_blockchain_rpc_calls_total",
			Help: "Total number of blockchain RPC calls",
		},
		[]string{"method", "status"},
	)

	// BlockchainRPCDuration RPC 调用延迟
	BlockchainRPCDuration = promauto.NewHistogramVec(
		prometheus.HistogramOpts{
			Name:    "chainsql_blockchain_rpc_duration_seconds",
			Help:    "Duration of blockchain RPC calls in seconds",
			Buckets: prometheus.DefBuckets,
		},
		[]string{"method"},
	)
)

// RecordEventProcessed 记录事件处理结果
func RecordEventProcessed(eventType string, success bool) {
	status := "success"
	if !success {
		status = "error"
	}
	EventsProcessed.WithLabelValues(eventType, status).Inc()
}

// RecordDatabaseOperation 记录数据库操作结果
func RecordDatabaseOperation(operation string, success bool) {
	status := "success"
	if !success {
		status = "error"
	}
	DatabaseOperations.WithLabelValues(operation, status).Inc()
}

// RecordRPCCall 记录 RPC 调用结果
func RecordRPCCall(method string, success bool) {
	status := "success"
	if !success {
		status = "error"
	}
	BlockchainRPCCalls.WithLabelValues(method, status).Inc()
}
