// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

interface ISQLSync {
    struct ColumnDef {
        string name;
        string sqlType;
        bool isPrimaryKey;
    }

    // --- Events ---
    event TableCreated(string tableName, ColumnDef[] columns);
    event TableDropped(string tableName);
    event TableAltered(string tableName, string rawSql);
    event DataInserted(
        string indexed tableName,
        string[] columns,
        string[] values
    );
    event DataUpdated(
        string indexed tableName,
        string[] setColumns,
        string[] setValues,
        string whereClause
    );
    event DataDeleted(string indexed tableName, string whereClause);

    // Advanced Events
    event DataBatchInserted(
        string indexed tableName,
        string[] columns,
        string[][] values
    );
    event DataUpserted(
        string indexed tableName,
        string[] columns,
        string[] values,
        string conflictColumn
    );
    event IndexCreated(
        string indexed tableName,
        string indexName,
        string[] columns,
        bool isUnique
    );
    event TableTruncated(string indexed tableName);

    // --- DDL Functions ---
    function createTable(
        string calldata tableName,
        ColumnDef[] calldata columns
    ) external;

    function dropTable(string calldata tableName) external;

    function alterTable(
        string calldata tableName,
        string calldata rawSql
    ) external;

    function createIndex(
        string calldata tableName,
        string calldata indexName,
        string[] calldata columns,
        bool isUnique
    ) external;

    function truncateTable(string calldata tableName) external;

    // --- DML Functions ---
    function insert(
        string calldata tableName,
        string[] calldata columns,
        string[] calldata values
    ) external;

    function update(
        string calldata tableName,
        string[] calldata setColumns,
        string[] calldata setValues,
        string calldata whereClause
    ) external;

    function deleteData(
        string calldata tableName,
        string calldata whereClause
    ) external;

    function batchInsert(
        string calldata tableName,
        string[] calldata columns,
        string[] calldata flatValues
    ) external;

    function upsert(
        string calldata tableName,
        string[] calldata columns,
        string[] calldata values,
        string calldata conflictColumn
    ) external;

    // 移除 runRawSQL
}
