#!/bin/bash

# ChainSQL 快速启动脚本

set -e

echo "🚀 ChainSQL Quick Start Script"
echo "================================"
echo ""

# 检查依赖
check_dependency() {
    if ! command -v $1 &> /dev/null; then
        echo "❌ $1 is not installed"
        echo "Please install $1 first"
        exit 1
    fi
}

echo "📋 Checking dependencies..."
check_dependency "node"
check_dependency "npm"
check_dependency "go"
check_dependency "docker"
echo "✅ All dependencies found"
echo ""

# 1. 安装 Hardhat 依赖
if [ ! -d "hardhat/node_modules" ]; then
    echo "📦 Installing Hardhat dependencies..."
    cd hardhat
    npm install
    cd ..
    echo "✅ Hardhat dependencies installed"
else
    echo "✅ Hardhat dependencies already installed"
fi
echo ""

# 2. 启动 PostgreSQL（如果使用 Docker）
echo "🐘 Checking PostgreSQL..."
if docker ps | grep -q chainsql-db; then
    echo "✅ PostgreSQL is already running"
else
    echo "Starting PostgreSQL with Docker Compose..."
    docker-compose up -d postgres
    echo "Waiting for PostgreSQL to be ready..."
    sleep 5
    echo "✅ PostgreSQL started"
fi
echo ""

# # 3. 启动 Hardhat 节点（后台）
# echo "⛓️  Starting Hardhat node..."
# cd hardhat
# npx hardhat node > ../hardhat-node.log 2>&1 &
# HARDHAT_PID=$!
# echo $HARDHAT_PID > ../hardhat.pid
# cd ..
# echo "✅ Hardhat node started (PID: $HARDHAT_PID)"
# echo "   Logs: hardhat-node.log"
# echo "   Waiting for node to be ready..."
# sleep 3
# echo ""

# 4. 部署合约
echo "📝 Deploying contracts..."
cd hardhat
npm run deploy > ../deploy.log 2>&1
cd ..
echo "✅ Contracts deployed"
echo "   Deployment info: hardhat/deployments/localhost.json"
echo ""

# 5. 更新 ChainSQL 配置
echo "⚙️  Updating ChainSQL configuration..."
FACTORY_ADDRESS=$(cat hardhat/deployments/localhost.json | grep -o '"SQLSyncFactory": "[^"]*"' | cut -d'"' -f4)

cat > configs/config.yaml << EOF
chain:
  rpc_url: "http://localhost:8545"
  factory_address: "$FACTORY_ADDRESS"
  chain_id: 31337

database:
  dsn: "postgres://chainsql:password123@localhost:5432/chainsql?sslmode=disable"

sync:
  start_block: 0
  confirmations: 1
  poll_interval: "1s"
  max_batch_size: 100
  address_batch_size: 50

log:
  level: "debug"
  development: true

http:
  port: 8080
EOF

echo "✅ Configuration updated"
echo ""

# # 6. 启动 ChainSQL
# echo "🔄 Starting ChainSQL..."
# make build
# ./bin/chainsql > chainsql.log 2>&1 &
# CHAINSQL_PID=$!
# echo $CHAINSQL_PID > chainsql.pid
# echo "✅ ChainSQL started (PID: $CHAINSQL_PID)"
# echo "   Logs: chainsql.log"
# echo ""

# 7. 等待服务就绪
echo "⏳ Waiting for services to be ready..."
sleep 3

# 检查健康状态
if curl -s http://localhost:8080/health > /dev/null; then
    echo "✅ ChainSQL is healthy"
else
    echo "⚠️  ChainSQL health check failed, check logs"
fi
echo ""

# 8. 显示状态
echo "================================"
echo "🎉 All services started!"
echo ""
echo "📊 Service Status:"
echo "  - Hardhat Node:  http://localhost:8545"
echo "  - ChainSQL HTTP: http://localhost:8080"
echo "  - PostgreSQL:    localhost:5432"
echo ""
echo "📝 Useful Commands:"
echo "  - View ChainSQL logs:  tail -f chainsql.log"
echo "  - View Hardhat logs:   tail -f hardhat-node.log"
echo "  - Check metrics:       curl http://localhost:8080/metrics"
echo "  - Stop all services:   ./scripts/stop.sh"
echo ""
echo "🧪 Run Integration Test:"
echo "  cd hardhat && node scripts/integration-test.js"
echo ""
echo "🗄️  Query Database:"
echo "  docker exec -it chainsql-db psql -U chainsql -d chainsql -c \"SELECT * FROM _chainsql_instances;\""
echo ""
