const hre = require("hardhat");

async function main() {
  console.log("🚀 Starting deployment...\n");

  // 获取部署账户
  const [deployer] = await hre.ethers.getSigners();
  console.log("📝 Deploying contracts with account:", deployer.address);
  const balance = await hre.ethers.provider.getBalance(deployer.address);
  console.log("💰 Account balance:", hre.ethers.formatEther(balance), "ETH\n");

  // 部署 SQLSyncFactory
  console.log("📦 Deploying SQLSyncFactory...");
  const SQLSyncFactory = await hre.ethers.getContractFactory("SQLSyncFactory");
  const factory = await SQLSyncFactory.deploy();
  await factory.waitForDeployment();
  const factoryAddress = await factory.getAddress();
  console.log("✅ SQLSyncFactory deployed to:", factoryAddress);

  // 创建一个测试实例
  console.log("\n🏭 Creating test instance...");
  const tx = await factory.createInstance();
  const receipt = await tx.wait();

  // 从事件中获取实例地址 (ethers v6)
  const event = receipt.logs.find(log => {
    try {
      const parsed = factory.interface.parseLog(log);
      return parsed && parsed.name === "InstanceCreated";
    } catch {
      return false;
    }
  });

  const parsedEvent = factory.interface.parseLog(event);
  const instanceAddress = parsedEvent.args.instance;

  console.log("✅ Test instance created at:", instanceAddress);
  console.log("👤 Instance owner:", parsedEvent.args.owner);

  // 保存部署信息
  const network = await hre.ethers.provider.getNetwork();
  const deploymentInfo = {
    network: hre.network.name,
    chainId: Number(network.chainId), // 转换 BigInt 为 Number
    deployer: deployer.address,
    contracts: {
      SQLSyncFactory: factoryAddress,
      testInstance: instanceAddress
    },
    timestamp: new Date().toISOString()
  };

  console.log("\n📄 Deployment Summary:");
  console.log(JSON.stringify(deploymentInfo, null, 2));

  // 保存到文件
  const fs = require("fs");
  const path = require("path");
  const deploymentPath = path.join(__dirname, "../deployments");

  if (!fs.existsSync(deploymentPath)) {
    fs.mkdirSync(deploymentPath, { recursive: true });
  }

  fs.writeFileSync(
    path.join(deploymentPath, `${hre.network.name}.json`),
    JSON.stringify(deploymentInfo, null, 2)
  );

  console.log("\n✅ Deployment info saved to deployments/" + hre.network.name + ".json");

  // 更新 ChainSQL 配置
  console.log("\n📝 Update your ChainSQL config with:");
  console.log(`
chain:
  rpc_url: "http://localhost:8545"
  factory_address: "${factoryAddress}"
  chain_id: 31337
  `);
}

main()
  .then(() => process.exit(0))
  .catch((error) => {
    console.error(error);
    process.exit(1);
  });
