const hre = require("hardhat");
const fs = require("fs");
const path = require("path");

async function main() {
    console.log("🎯 ChainSQL Integration Test\n");

    // 读取部署信息
    const deploymentPath = path.join(__dirname, "../deployments/localhost.json");
    if (!fs.existsSync(deploymentPath)) {
        console.error("❌ No deployment found. Please run: npm run deploy");
        process.exit(1);
    }

    const deployment = JSON.parse(fs.readFileSync(deploymentPath, "utf8"));
    console.log("📄 Using deployment:", deployment.contracts);

    const [owner] = await hre.ethers.getSigners();
    const sqlSync = await hre.ethers.getContractAt("SQLSync", deployment.contracts.testInstance);

    console.log("\n1️⃣ Creating table 'products'...");
    let tx = await sqlSync.createTable("products", [
        { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
        { name: "name", sqlType: "VARCHAR(100)", isPrimaryKey: false },
        { name: "price", sqlType: "DECIMAL(10,2)", isPrimaryKey: false },
        { name: "stock", sqlType: "INTEGER", isPrimaryKey: false },
        { name: "status", sqlType: "VARCHAR(20)", isPrimaryKey: false }
    ]);
    await tx.wait();
    console.log("✅ Table created", tx.hash);

    console.log("\n2️⃣ Inserting product...");
    tx = await sqlSync.insert("products",
        ["id", "name", "price", "stock", "status"],
        ["1", "Laptop", "999.99", "10", "available"]
    );
    await tx.wait();
    console.log("✅ Product inserted");

    console.log("\n3️⃣ Batch inserting products...");
    tx = await sqlSync.batchInsert("products",
        ["id", "name", "price", "stock", "status"],
        [
            "2", "Mouse", "29.99", "50", "available",
            "3", "Keyboard", "79.99", "30", "available",
            "4", "Monitor", "299.99", "15", "available",
            "5", "Headphones", "149.99", "0", "out_of_stock"
        ]
    );
    await tx.wait();
    console.log("✅ Batch insert completed");

    console.log("\n4️⃣ Updating product status...");
    tx = await sqlSync.update("products",
        ["status"],
        ["low_stock"],
        "stock < 20 AND status = 'available'"
    );
    await tx.wait();
    console.log("✅ Product status updated");

    console.log("\n5️⃣ Updating price...");
    tx = await sqlSync.update("products",
        ["price"],
        ["899.99"],
        "id = 1"
    );
    await tx.wait();
    console.log("✅ Price updated");

    console.log("\n6️⃣ Deleting out of stock products...");
    tx = await sqlSync.deleteData("products", "status = 'out_of_stock'");
    await tx.wait();
    console.log("✅ Products deleted");

    console.log("\n7️⃣ Creating index...");
    tx = await sqlSync.createIndex("products", "idx_status", ["status"], false);
    await tx.wait();
    console.log("✅ Index created");

    console.log("\n✅ Integration test completed!");
    console.log("\n📊 Check your PostgreSQL database:");
    console.log(`   SELECT * FROM "${deployment.contracts.testInstance}".products;`);
    console.log("\n💡 ChainSQL should have synced all these operations to the database.");
}

main()
    .then(() => process.exit(0))
    .catch((error) => {
        console.error(error);
        process.exit(1);
    });
