package executor

import (
	"context"
	"fmt"
	"log"
	"strings"

	"chain-sql/internal/database"
)

// SQL 类型白名单 (PostgreSQL 常用类型)
// 凡是不在这个列表里的类型定义，HandleTableCreated 都会拒绝执行
var allowedSqlTypes = map[string]bool{
	"int": true, "integer": true, "bigint": true, "smallint": true,
	"text": true, "varchar": true, "char": true, "character varying": true,
	"boolean": true, "bool": true,
	"decimal": true, "numeric": true, "real": true, "double precision": true,
	"timestamp": true, "date": true, "time": true,
	"json": true, "jsonb": true,
	"bytea": true,
}

type Executor struct {
	db *database.DB
}

func NewExecutor(db *database.DB) *Executor {
	return &Executor{db: db}
}

// validateSqlType 校验 SQL 类型是否合法
func validateSqlType(t string) bool {
	// 移除长度修饰符，例如 "VARCHAR(50)" -> "varchar"
	parts := strings.Split(t, "(")
	baseType := strings.TrimSpace(strings.ToLower(parts[0]))
	return allowedSqlTypes[baseType]
}

// EnsureSchema 为每个合约创建独立的 Schema
func (e *Executor) EnsureSchema(ctx context.Context, contractAddr string) error {
	schemaName := QuoteIdentifier(contractAddr)
	sql := fmt.Sprintf(`CREATE SCHEMA IF NOT EXISTS %s`, schemaName)
	return e.db.Exec(ctx, sql)
}

// HandleTableCreated 处理建表事件 (含安全校验)
func (e *Executor) HandleTableCreated(ctx context.Context, contractAddr string, tableName string, columns []struct {
	Name         string
	SqlType      string
	IsPrimaryKey bool
}) error {
	schemaName := QuoteIdentifier(contractAddr)
	safeTableName := QuoteIdentifier(tableName)

	var colDefs []string
	var pks []string

	for _, col := range columns {
		// --- 安全检查核心逻辑 ---
		if !validateSqlType(col.SqlType) {
			return fmt.Errorf("security error: invalid or forbidden sql type '%s' for column '%s'", col.SqlType, col.Name)
		}
		// ---------------------

		def := fmt.Sprintf("%s %s", QuoteIdentifier(col.Name), col.SqlType)
		colDefs = append(colDefs, def)

		if col.IsPrimaryKey {
			pks = append(pks, QuoteIdentifier(col.Name))
		}
	}

	if len(pks) > 0 {
		pkDef := fmt.Sprintf("PRIMARY KEY (%s)", strings.Join(pks, ", "))
		colDefs = append(colDefs, pkDef)
	}

	fullSql := fmt.Sprintf(
		`CREATE TABLE IF NOT EXISTS %s.%s (%s)`,
		schemaName,
		safeTableName,
		strings.Join(colDefs, ", "),
	)

	log.Printf("Executing DDL: %s", fullSql)
	return e.db.Exec(ctx, fullSql)
}

// HandleDataInserted 处理插入事件
func (e *Executor) HandleDataInserted(ctx context.Context, contractAddr string, tableName string, columns []string, values []string) error {
	schemaName := QuoteIdentifier(contractAddr)
	safeTableName := QuoteIdentifier(tableName)

	// 构建占位符 ($1, $2...)
	placeholders := make([]string, len(values))
	args := make([]interface{}, len(values))
	safeCols := make([]string, len(columns))

	for i, v := range values {
		placeholders[i] = fmt.Sprintf("$%d", i+1)
		args[i] = v
	}
	for i, c := range columns {
		safeCols[i] = QuoteIdentifier(c)
	}

	sql := fmt.Sprintf(
		`INSERT INTO %s.%s (%s) VALUES (%s)`,
		schemaName,
		safeTableName,
		strings.Join(safeCols, ", "),
		strings.Join(placeholders, ", "),
	)

	return e.db.Exec(ctx, sql, args...)
}

// QuoteIdentifier 对标识符加双引号，防止 SQL 注入和关键字冲突
func QuoteIdentifier(s string) string {
	return fmt.Sprintf(`"%s"`, strings.ReplaceAll(s, `"`, `""`))
}

// HandleDataUpdated 处理更新事件
func (e *Executor) HandleDataUpdated(ctx context.Context, contractAddr string, tableName string, setColumns []string, setValues []string, whereClause string) error {
	if len(setColumns) != len(setValues) {
		return fmt.Errorf("columns and values length mismatch: %d vs %d", len(setColumns), len(setValues))
	}

	schemaName := QuoteIdentifier(contractAddr)
	safeTableName := QuoteIdentifier(tableName)

	// 构建 SET 子句
	var setParts []string
	args := make([]interface{}, 0, len(setValues))

	for i, col := range setColumns {
		setParts = append(setParts, fmt.Sprintf("%s = $%d", QuoteIdentifier(col), i+1))
		args = append(args, setValues[i])
	}

	// 解析 WHERE 子句
	// 首先需要获取表的列信息来构建白名单
	// 为了简化，这里使用所有 setColumns 作为允许的列（实际应该查询表结构）
	allowedCols := append([]string{}, setColumns...)
	parser := NewWhereParser(allowedCols)

	whereCond, err := parser.Parse(whereClause)
	if err != nil {
		return fmt.Errorf("parse WHERE clause failed: %w", err)
	}

	// 调整参数占位符编号
	adjustedWhereSQL := whereCond.SQL
	for i := len(args); i > 0; i-- {
		placeholder := fmt.Sprintf("$%d", i-len(args)+len(setValues))
		adjustedWhereSQL = strings.Replace(adjustedWhereSQL, fmt.Sprintf("$%d", i), placeholder, -1)
	}

	// 追加 WHERE 参数
	args = append(args, whereCond.Args...)

	sql := fmt.Sprintf(
		`UPDATE %s.%s SET %s WHERE %s`,
		schemaName,
		safeTableName,
		strings.Join(setParts, ", "),
		adjustedWhereSQL,
	)

	log.Printf("Executing UPDATE: %s", sql)
	return e.db.Exec(ctx, sql, args...)
}

// HandleDataDeleted 处理删除事件
func (e *Executor) HandleDataDeleted(ctx context.Context, contractAddr string, tableName string, whereClause string) error {
	schemaName := QuoteIdentifier(contractAddr)
	safeTableName := QuoteIdentifier(tableName)

	// 解析 WHERE 子句
	// 注意：这里需要表的列信息，简化处理，允许常见列名
	// 生产环境应该查询 information_schema 获取实际列
	parser := NewWhereParser([]string{"id", "name", "status", "email", "created_at", "updated_at"})

	whereCond, err := parser.Parse(whereClause)
	if err != nil {
		return fmt.Errorf("parse WHERE clause failed: %w", err)
	}

	if whereCond.SQL == "" {
		return fmt.Errorf("DELETE without WHERE clause is not allowed for safety")
	}

	sql := fmt.Sprintf(
		`DELETE FROM %s.%s WHERE %s`,
		schemaName,
		safeTableName,
		whereCond.SQL,
	)

	log.Printf("Executing DELETE: %s", sql)
	return e.db.Exec(ctx, sql, whereCond.Args...)
}

// HandleBatchInserted 处理批量插入事件
func (e *Executor) HandleBatchInserted(ctx context.Context, contractAddr string, tableName string, columns []string, rows [][]string) error {
	if len(rows) == 0 {
		return nil
	}

	schemaName := QuoteIdentifier(contractAddr)
	safeTableName := QuoteIdentifier(tableName)
	safeCols := make([]string, len(columns))

	for i, c := range columns {
		safeCols[i] = QuoteIdentifier(c)
	}

	// 构建批量插入 SQL
	var valueParts []string
	args := make([]interface{}, 0, len(rows)*len(columns))
	paramIndex := 1

	for _, row := range rows {
		if len(row) != len(columns) {
			return fmt.Errorf("row length mismatch: expected %d, got %d", len(columns), len(row))
		}

		placeholders := make([]string, len(row))
		for i, val := range row {
			placeholders[i] = fmt.Sprintf("$%d", paramIndex)
			args = append(args, val)
			paramIndex++
		}
		valueParts = append(valueParts, fmt.Sprintf("(%s)", strings.Join(placeholders, ", ")))
	}

	sql := fmt.Sprintf(
		`INSERT INTO %s.%s (%s) VALUES %s`,
		schemaName,
		safeTableName,
		strings.Join(safeCols, ", "),
		strings.Join(valueParts, ", "),
	)

	log.Printf("Executing BATCH INSERT: %d rows into %s.%s", len(rows), contractAddr, tableName)
	return e.db.Exec(ctx, sql, args...)
}
