# ChainSQL 使用示例

## 快速开始

### 1. 启动服务

```bash
# 使用 Docker Compose（推荐）
docker-compose up -d

# 或本地运行
make run
```

### 2. 部署智能合约

```solidity
// 部署 SQLSyncFactory
const factory = await SQLSyncFactory.deploy();

// 创建实例
const tx = await factory.createInstance();
const receipt = await tx.wait();
const instanceAddr = receipt.events[0].args.instance;
```

### 3. 使用合约操作数据库

```javascript
const sqlSync = await ethers.getContractAt("SQLSync", instanceAddr);

// 创建表
await sqlSync.createTable("users", [
  { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
  { name: "name", sqlType: "VARCHAR(100)", isPrimaryKey: false },
  { name: "email", sqlType: "VARCHAR(255)", isPrimaryKey: false },
  { name: "status", sqlType: "VARCHAR(20)", isPrimaryKey: false }
]);

// 插入数据
await sqlSync.insert("users", 
  ["id", "name", "email", "status"],
  ["1", "Alice", "alice@example.com", "active"]
);

// 更新数据
await sqlSync.update("users",
  ["status"],
  ["inactive"],
  "id = 1"
);

// 删除数据
await sqlSync.deleteData("users", "status = 'inactive'");

// 批量插入
await sqlSync.batchInsert("users",
  ["id", "name", "email", "status"],
  [
    "2", "Bob", "bob@example.com", "active",
    "3", "Charlie", "charlie@example.com", "active"
  ]
);
```

### 4. 查询数据库

ChainSQL 会自动将事件同步到 PostgreSQL，你可以直接查询：

```sql
-- 连接数据库
psql -U chainsql -d chainsql

-- 查询数据（每个合约有独立的 schema）
SELECT * FROM "0x1234...abcd".users WHERE status = 'active';

-- 查看所有活跃合约
SELECT * FROM _chainsql_instances WHERE status = 'active';

-- 查看同步状态
SELECT * FROM _chainsql_cursor;
```

### 5. 监控服务

```bash
# 健康检查
curl http://localhost:8080/health

# 就绪检查
curl http://localhost:8080/ready

# Prometheus 指标
curl http://localhost:8080/metrics
```

## 配置示例

### 开发环境

```yaml
# configs/config.yaml
chain:
  rpc_url: "http://localhost:8545"
  factory_address: "0x5FbDB2315678afecb367f032d93F642f64180aa3"
  chain_id: 31337

database:
  dsn: "postgres://chainsql:password123@localhost:5432/chainsql?sslmode=disable"

sync:
  start_block: 0
  confirmations: 1
  poll_interval: "1s"
  max_batch_size: 100
  address_batch_size: 50

log:
  level: "debug"
  development: true

http:
  port: 8080
```

### 生产环境

```yaml
# configs/config.yaml
chain:
  rpc_url: "https://mainnet.infura.io/v3/YOUR_API_KEY"
  factory_address: "0x..."
  chain_id: 1

database:
  dsn: "postgres://user:password@db.example.com:5432/chainsql?sslmode=require"

sync:
  start_block: 18000000
  confirmations: 12
  poll_interval: "5s"
  max_batch_size: 1000
  address_batch_size: 100

log:
  level: "info"
  development: false

http:
  port: 8080
```

## WHERE 子句示例

ChainSQL 支持安全的 WHERE 子句解析：

```javascript
// 简单条件
await sqlSync.update("users", ["status"], ["active"], "id = 1");

// 多条件
await sqlSync.deleteData("users", "status = 'inactive' AND created_at < '2024-01-01'");

// IN 操作
await sqlSync.update("users", ["verified"], ["true"], "id IN (1, 2, 3)");

// LIKE 操作
await sqlSync.deleteData("users", "email LIKE '%@spam.com'");

// IS NULL
await sqlSync.deleteData("users", "email IS NULL");

// 复杂条件
await sqlSync.update("users", 
  ["status"], 
  ["premium"], 
  "score > 100 AND status = 'active'"
);
```

## 监控集成

### Prometheus 配置

```yaml
# prometheus.yml
scrape_configs:
  - job_name: 'chainsql'
    static_configs:
      - targets: ['localhost:8080']
```

### Grafana 仪表板

关键指标：
- `chainsql_events_processed_total` - 事件处理总数
- `chainsql_sync_block_height` - 同步进度
- `chainsql_active_contracts` - 活跃合约数
- `chainsql_database_operations_total` - 数据库操作数

## 故障排查

### 检查日志

```bash
# Docker
docker-compose logs -f chainsql

# 本地
tail -f logs/chainsql.log
```

### 常见问题

**Q: 服务启动失败**
```bash
# 检查数据库连接
curl http://localhost:8080/ready

# 查看详细日志
docker-compose logs chainsql
```

**Q: 事件未同步**
```bash
# 检查同步状态
psql -U chainsql -c "SELECT * FROM _chainsql_cursor;"

# 检查合约注册
psql -U chainsql -c "SELECT * FROM _chainsql_instances;"
```

**Q: 性能问题**
```bash
# 查看 Prometheus 指标
curl http://localhost:8080/metrics | grep duration
```

## 最佳实践

1. **安全性**
   - 始终使用参数化查询
   - 验证 WHERE 子句的列名
   - 定期审查数据库权限

2. **性能**
   - 使用批量插入代替单条插入
   - 合理设置 `max_batch_size`
   - 监控数据库连接池

3. **可靠性**
   - 设置合适的 `confirmations` 值
   - 定期备份数据库
   - 监控同步延迟

4. **监控**
   - 配置 Prometheus 告警
   - 监控关键指标
   - 定期检查健康状态
