# Hardhat 测试环境

本目录包含 ChainSQL 项目的 Hardhat 测试环境。

## 🚀 快速开始

### 1. 安装依赖

```bash
cd hardhat
npm install
```

### 2. 启动本地节点

```bash
# Terminal 1
npm run node
```

这将启动一个本地以太坊节点在 `http://localhost:8545`

### 3. 部署合约

```bash
# Terminal 2
npm run deploy
```

部署信息会保存在 `deployments/localhost.json`

### 4. 更新 ChainSQL 配置

根据部署输出，更新 `../configs/config.yaml`：

```yaml
chain:
  rpc_url: "http://localhost:8545"
  factory_address: "0x5FbDB2315678afecb367f032d93F642f64180aa3"  # 使用实际地址
  chain_id: 31337
```

### 5. 启动 ChainSQL

```bash
# Terminal 3
cd ..
make run
```

### 6. 运行集成测试

```bash
# Terminal 2
cd hardhat
node scripts/integration-test.js
```

## 📋 可用命令

```bash
npm run node          # 启动本地节点
npm run compile       # 编译合约
npm run test          # 运行单元测试
npm run deploy        # 部署合约
npm run clean         # 清理编译产物
```

## 🧪 测试

### 运行单元测试

```bash
npm test
```

测试覆盖：
- ✅ Factory 合约功能
- ✅ 表创建和管理
- ✅ 数据插入、更新、删除
- ✅ 批量操作
- ✅ 权限控制
- ✅ 完整工作流

### 运行集成测试

```bash
node scripts/integration-test.js
```

这将执行一个完整的工作流，包括：
1. 创建表
2. 插入数据
3. 批量插入
4. 更新数据
5. 删除数据
6. 创建索引

## 📁 目录结构

```
hardhat/
├── contracts/          # Solidity 合约
│   ├── ISQLSync.sol
│   ├── SQLSync.sol
│   └── SQLSyncFactory.sol
├── scripts/           # 部署和测试脚本
│   ├── deploy.js
│   └── integration-test.js
├── test/              # 单元测试
│   └── ChainSQL.test.js
├── deployments/       # 部署信息（自动生成）
├── hardhat.config.js  # Hardhat 配置
└── package.json       # 依赖配置
```

## 🔧 配置

### Hardhat 网络配置

```javascript
// hardhat.config.js
networks: {
  hardhat: {
    chainId: 31337,
    mining: {
      auto: true,      // 自动挖矿
      interval: 0      // 立即出块
    }
  }
}
```

### 账户

Hardhat 默认提供 10 个测试账户，每个账户有 10000 ETH。

查看账户：
```bash
npx hardhat accounts
```

## 💡 使用技巧

### 1. 查看合约事件

```javascript
const receipt = await tx.wait();
console.log(receipt.events);
```

### 2. 使用 console.log 调试

在 Solidity 中：
```solidity
import "hardhat/console.sol";

console.log("Value:", value);
```

### 3. 时间旅行

```javascript
await network.provider.send("evm_increaseTime", [3600]); // 前进 1 小时
await network.provider.send("evm_mine"); // 挖一个块
```

### 4. 快照和恢复

```javascript
const snapshot = await network.provider.send("evm_snapshot");
// ... 执行操作
await network.provider.send("evm_revert", [snapshot]);
```

## 🐛 故障排查

### 问题：端口已被占用

```bash
# 查找占用端口的进程
lsof -i :8545

# 杀死进程
kill -9 <PID>
```

### 问题：合约部署失败

```bash
# 清理并重新编译
npm run clean
npm run compile
```

### 问题：测试失败

```bash
# 重启节点
# Ctrl+C 停止节点，然后重新启动
npm run node
```

## 📚 参考资料

- [Hardhat 文档](https://hardhat.org/docs)
- [Ethers.js 文档](https://docs.ethers.org/)
- [Chai 断言库](https://www.chaijs.com/)
