const { expect } = require("chai");
const { ethers } = require("hardhat");

describe("ChainSQL Contracts", function () {
    let factory;
    let sqlSync;
    let owner;
    let addr1;
    let addr2;

    beforeEach(async function () {
        [owner, addr1, addr2] = await ethers.getSigners();

        // 部署 Factory
        const SQLSyncFactory = await ethers.getContractFactory("SQLSyncFactory");
        factory = await SQLSyncFactory.deploy();
        await factory.deployed();

        // 创建实例
        const tx = await factory.createInstance();
        const receipt = await tx.wait();
        const event = receipt.events?.find(e => e.event === "InstanceCreated");
        const instanceAddress = event?.args?.instance;

        // 获取实例合约
        sqlSync = await ethers.getContractAt("SQLSync", instanceAddress);
    });

    describe("SQLSyncFactory", function () {
        it("Should create a new instance", async function () {
            const tx = await factory.createInstance();
            const receipt = await tx.wait();
            const event = receipt.events?.find(e => e.event === "InstanceCreated");

            expect(event).to.not.be.undefined;
            expect(event.args.owner).to.equal(owner.address);
            expect(event.args.instance).to.be.properAddress;
        });

        it("Should track instance count", async function () {
            const initialCount = await factory.getInstanceCount();
            await factory.createInstance();
            const newCount = await factory.getInstanceCount();

            expect(newCount).to.equal(initialCount.add(1));
        });

        it("Should mark instances as valid", async function () {
            const tx = await factory.createInstance();
            const receipt = await tx.wait();
            const event = receipt.events?.find(e => e.event === "InstanceCreated");
            const instanceAddr = event.args.instance;

            expect(await factory.isValidInstance(instanceAddr)).to.be.true;
        });
    });

    describe("SQLSync - Table Operations", function () {
        it("Should create a table", async function () {
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
                { name: "name", sqlType: "VARCHAR(100)", isPrimaryKey: false },
                { name: "email", sqlType: "VARCHAR(255)", isPrimaryKey: false }
            ];

            await expect(sqlSync.createTable("users", columns))
                .to.emit(sqlSync, "TableCreated")
                .withArgs("users", columns);
        });

        it("Should only allow owner to create tables", async function () {
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true }
            ];

            await expect(
                sqlSync.connect(addr1).createTable("users", columns)
            ).to.be.revertedWith("Not owner");
        });
    });

    describe("SQLSync - Data Operations", function () {
        beforeEach(async function () {
            // 先创建表
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
                { name: "name", sqlType: "VARCHAR(100)", isPrimaryKey: false },
                { name: "status", sqlType: "VARCHAR(20)", isPrimaryKey: false }
            ];
            await sqlSync.createTable("users", columns);
        });

        it("Should insert data", async function () {
            const columns = ["id", "name", "status"];
            const values = ["1", "Alice", "active"];

            await expect(sqlSync.insert("users", columns, values))
                .to.emit(sqlSync, "DataInserted")
                .withArgs("users", columns, values);
        });

        it("Should update data", async function () {
            const setColumns = ["status"];
            const setValues = ["inactive"];
            const whereClause = "id = 1";

            await expect(sqlSync.update("users", setColumns, setValues, whereClause))
                .to.emit(sqlSync, "DataUpdated")
                .withArgs("users", setColumns, setValues, whereClause);
        });

        it("Should delete data", async function () {
            const whereClause = "status = 'inactive'";

            await expect(sqlSync.deleteData("users", whereClause))
                .to.emit(sqlSync, "DataDeleted")
                .withArgs("users", whereClause);
        });

        it("Should batch insert data", async function () {
            const columns = ["id", "name", "status"];
            const flatValues = [
                "2", "Bob", "active",
                "3", "Charlie", "active",
                "4", "David", "pending"
            ];

            await expect(sqlSync.batchInsert("users", columns, flatValues))
                .to.emit(sqlSync, "DataBatchInserted");
        });

        it("Should only allow owner to modify data", async function () {
            await expect(
                sqlSync.connect(addr1).insert("users", ["id"], ["1"])
            ).to.be.revertedWith("Not owner");
        });
    });

    describe("SQLSync - Advanced Operations", function () {
        it("Should create index", async function () {
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
                { name: "email", sqlType: "VARCHAR(255)", isPrimaryKey: false }
            ];
            await sqlSync.createTable("users", columns);

            await expect(
                sqlSync.createIndex("users", "idx_email", ["email"], false)
            ).to.emit(sqlSync, "IndexCreated")
                .withArgs("users", "idx_email", ["email"], false);
        });

        it("Should truncate table", async function () {
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true }
            ];
            await sqlSync.createTable("users", columns);

            await expect(sqlSync.truncateTable("users"))
                .to.emit(sqlSync, "TableTruncated")
                .withArgs("users");
        });

        it("Should upsert data", async function () {
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
                { name: "name", sqlType: "VARCHAR(100)", isPrimaryKey: false }
            ];
            await sqlSync.createTable("users", columns);

            const cols = ["id", "name"];
            const vals = ["1", "Alice"];
            const conflictCol = "id";

            await expect(sqlSync.upsert("users", cols, vals, conflictCol))
                .to.emit(sqlSync, "DataUpserted")
                .withArgs("users", cols, vals, conflictCol);
        });
    });

    describe("Integration Test", function () {
        it("Should handle complete workflow", async function () {
            // 1. 创建表
            const columns = [
                { name: "id", sqlType: "INTEGER", isPrimaryKey: true },
                { name: "username", sqlType: "VARCHAR(50)", isPrimaryKey: false },
                { name: "email", sqlType: "VARCHAR(255)", isPrimaryKey: false },
                { name: "status", sqlType: "VARCHAR(20)", isPrimaryKey: false }
            ];
            await sqlSync.createTable("users", columns);

            // 2. 插入数据
            await sqlSync.insert("users",
                ["id", "username", "email", "status"],
                ["1", "alice", "alice@example.com", "active"]
            );

            // 3. 批量插入
            await sqlSync.batchInsert("users",
                ["id", "username", "email", "status"],
                [
                    "2", "bob", "bob@example.com", "active",
                    "3", "charlie", "charlie@example.com", "pending"
                ]
            );

            // 4. 更新数据
            await sqlSync.update("users",
                ["status"],
                ["verified"],
                "id = 1"
            );

            // 5. 删除数据
            await sqlSync.deleteData("users", "status = 'pending'");

            // 验证所有操作都成功（通过事件）
            // 实际测试中，ChainSQL 会监听这些事件并同步到数据库
        });
    });
});
