package config

import (
	"strings"
	"time"

	"github.com/spf13/viper"
)

type Config struct {
	Chain    ChainConfig    `mapstructure:"chain"`
	Database DatabaseConfig `mapstructure:"database"`
	Sync     SyncConfig     `mapstructure:"sync"`
	Log      LogConfig      `mapstructure:"log"`
	HTTP     HTTPConfig     `mapstructure:"http"`
}

type ChainConfig struct {
	RPCURL         string `mapstructure:"rpc_url"`
	FactoryAddress string `mapstructure:"factory_address"`
	ChainID        int64  `mapstructure:"chain_id"`
}

type DatabaseConfig struct {
	DSN string `mapstructure:"dsn"`
}

type SyncConfig struct {
	StartBlock       uint64        `mapstructure:"start_block"`        // 从哪个区块开始同步
	Confirmations    uint64        `mapstructure:"confirmations"`      // 确认区块数
	PollInterval     time.Duration `mapstructure:"poll_interval"`      // 轮询间隔
	MaxBatchSize     int           `mapstructure:"max_batch_size"`     // 批量大小
	AddressBatchSize int           `mapstructure:"address_batch_size"` // 地址批量大小
}

type LogConfig struct {
	Level       string `mapstructure:"level"`
	Development bool   `mapstructure:"development"`
}

type HTTPConfig struct {
	Port int `mapstructure:"port"`
}

func Load(path string) (*Config, error) {
	v := viper.New()
	v.SetConfigFile(path)
	v.AutomaticEnv()
	v.SetEnvKeyReplacer(strings.NewReplacer(".", "_"))

	if err := v.ReadInConfig(); err != nil {
		return nil, err
	}

	var cfg Config
	if err := v.Unmarshal(&cfg); err != nil {
		return nil, err
	}
	return &cfg, nil
}
