# ChainSQL

[![Go Version](https://img.shields.io/badge/Go-1.24+-00ADD8?style=flat&logo=go)](https://golang.org)
[![License](https://img.shields.io/badge/License-MIT-blue.svg)](LICENSE)

**ChainSQL** 是一个区块链到 SQL 数据库的同步服务，它监听以太坊智能合约发出的事件，并将这些事件转换为实际的数据库操作。

## ✨ 核心特性

- 🔄 **自动同步**：实时监听区块链事件并同步到 PostgreSQL
- 🏭 **工厂模式**：支持动态创建和管理多个合约实例
- 🔒 **安全第一**：SQL 注入防护、WHERE 子句解析、类型白名单验证
- 📊 **完整 CRUD**：支持 CREATE TABLE、INSERT、UPDATE、DELETE 操作
- 🚀 **高性能**：批量插入优化、并发处理、连接池管理
- 📈 **可观测性**：Prometheus 监控指标、结构化日志、健康检查
- 🐳 **容器化**：Docker 和 Docker Compose 支持

## 🏗️ 架构概览

```
┌─────────────────┐
│  Smart Contract │  (Solidity)
│   SQLSync.sol   │
└────────┬────────┘
         │ Events (TableCreated, DataInserted, etc.)
         ▼
┌─────────────────┐
│  ChainSQL App   │  (Go)
│                 │
│  ┌───────────┐  │
│  │ Factory   │  │  监听新合约实例
│  │ Listener  │  │
│  └───────────┘  │
│                 │
│  ┌───────────┐  │
│  │   Data    │  │  监听数据事件
│  │ Listener  │  │
│  └───────────┘  │
│                 │
│  ┌───────────┐  │
│  │   SQL     │  │  执行数据库操作
│  │ Executor  │  │
│  └───────────┘  │
└────────┬────────┘
         ▼
┌─────────────────┐
│   PostgreSQL    │
└─────────────────┘
```

## 🚀 快速开始

### 前置要求

- Go 1.24+
- PostgreSQL 15+
- Docker & Docker Compose (可选)

### 本地开发

1. **克隆项目**
```bash
git clone <repository-url>
cd chain-sql
```

2. **配置文件**
```bash
cp configs/config.example.yaml configs/config.yaml
# 编辑 config.yaml 填入你的配置
```

3. **安装依赖**
```bash
go mod download
```

4. **运行服务**
```bash
make run
# 或者
go run cmd/main.go
```

### Docker 部署

```bash
# 构建并启动所有服务（PostgreSQL + ChainSQL）
make docker-up

# 查看日志
docker-compose logs -f chainsql

# 停止服务
make docker-down
```

## ⚙️ 配置说明

配置文件位于 `configs/config.yaml`：

```yaml
chain:
  rpc_url: "http://localhost:8545"      # 以太坊节点 RPC
  factory_address: "0x..."              # 工厂合约地址
  chain_id: 1337                        # 链 ID

database:
  dsn: "postgres://user:password@localhost:5432/chainsql?sslmode=disable"

sync:
  start_block: 0                        # 起始区块
  confirmations: 12                     # 确认区块数
  poll_interval: "2s"                   # 轮询间隔
  max_batch_size: 100                   # 单次查询最大区块数
  address_batch_size: 50                # 单次查询最大合约数

log:
  level: "info"                         # debug, info, warn, error
  development: false                    # 开发模式（彩色输出）

http:
  port: 8080                            # HTTP 服务端口
```

## 📡 HTTP 端点

ChainSQL 提供以下 HTTP 端点：

- `GET /metrics` - Prometheus 监控指标
- `GET /health` - 健康检查
- `GET /ready` - 就绪检查（包含数据库连接测试）

示例：
```bash
curl http://localhost:8080/health
curl http://localhost:8080/metrics
```

## 📊 监控指标

### 事件处理
- `chainsql_events_processed_total` - 事件处理总数（按类型和状态）
- `chainsql_event_processing_duration_seconds` - 事件处理延迟

### 同步状态
- `chainsql_sync_block_height` - 当前同步区块高度
- `chainsql_active_contracts` - 活跃合约数量

### 数据库操作
- `chainsql_database_operations_total` - 数据库操作总数
- `chainsql_database_operation_duration_seconds` - 数据库操作延迟

### 区块链 RPC
- `chainsql_blockchain_rpc_calls_total` - RPC 调用总数
- `chainsql_blockchain_rpc_duration_seconds` - RPC 调用延迟

## 🔐 安全特性

### SQL 注入防护

1. **WHERE 子句解析器**：安全解析 WHERE 条件，使用参数化查询
2. **类型白名单**：只允许预定义的 SQL 类型
3. **标识符转义**：自动转义表名和列名
4. **列名白名单**：验证 WHERE 子句中的列名

### 示例

```go
// ❌ 不安全（直接拼接）
sql := "DELETE FROM table WHERE " + whereClause

// ✅ 安全（使用解析器）
parser := NewWhereParser(allowedColumns)
cond, _ := parser.Parse("id = 123 AND status = 'active'")
sql := fmt.Sprintf("DELETE FROM table WHERE %s", cond.SQL)
db.Exec(ctx, sql, cond.Args...)
```

## 🛠️ 开发指南

### 项目结构

```
chain-sql/
├── cmd/                    # 应用入口
├── internal/
│   ├── api/               # HTTP 服务器
│   ├── blockchain/        # 区块链客户端
│   ├── config/            # 配置管理
│   ├── core/              # 核心业务逻辑
│   ├── database/          # 数据库操作
│   ├── executor/          # SQL 执行器
│   ├── logger/            # 结构化日志
│   └── metrics/           # 监控指标
├── contract/              # Solidity 合约
├── configs/               # 配置文件
└── docker-compose.yaml    # Docker 编排
```

### 添加新的事件处理

1. 在 `internal/core/data_listener.go` 添加事件结构体
2. 在 `internal/core/interfaces.go` 扩展 `EventHandler` 接口
3. 在 `internal/executor/sql_builder.go` 实现处理方法
4. 在 `processLog` 方法中添加 case 分支

### 运行测试

```bash
go test ./...
```

## 📝 智能合约

### SQLSync.sol

核心合约，提供 SQL 操作接口：

```solidity
// 创建表
function createTable(string calldata tableName, ColumnDef[] calldata columns) external;

// 插入数据
function insert(string calldata tableName, string[] calldata columns, string[] calldata values) external;

// 更新数据
function update(string calldata tableName, string[] calldata setColumns, string[] calldata setValues, string calldata whereClause) external;

// 删除数据
function deleteData(string calldata tableName, string calldata whereClause) external;
```

### SQLSyncFactory.sol

工厂合约，用于创建 SQLSync 实例：

```solidity
function createInstance() external returns (address instance);
```

## 🔄 工作流程

1. **部署合约**：部署 SQLSyncFactory 和 SQLSync 合约
2. **配置服务**：填写 `config.yaml` 中的合约地址和 RPC 端点
3. **启动服务**：运行 ChainSQL 应用
4. **创建实例**：调用工厂合约创建新的 SQLSync 实例
5. **执行操作**：通过合约调用 SQL 操作
6. **自动同步**：ChainSQL 监听事件并同步到数据库

## 🎯 使用场景

- 📊 **链上数据索引**：为 DApp 提供快速查询能力
- 🔍 **数据分析**：使用 SQL 分析区块链数据
- 📈 **报表生成**：基于链上数据生成业务报表
- 🏢 **多租户 SaaS**：每个用户一个合约实例

## 🤝 贡献

欢迎提交 Issue 和 Pull Request！

## 📄 许可证

MIT License

## 🙏 致谢

- [go-ethereum](https://github.com/ethereum/go-ethereum) - 以太坊 Go 客户端
- [pgx](https://github.com/jackc/pgx) - PostgreSQL 驱动
- [zap](https://github.com/uber-go/zap) - 高性能日志库
- [prometheus](https://prometheus.io/) - 监控系统
