package core

import (
	"context"
	"log"
	"sync"
	"time"

	"chain-sql/internal/database"

	"github.com/ethereum/go-ethereum/common"
)

type ContractManager struct {
	db        *database.DB
	contracts []common.Address
	mu        sync.RWMutex
}

func NewContractManager(db *database.DB) *ContractManager {
	return &ContractManager{
		db: db,
	}
}

// Reload 从数据库重新加载活跃合约列表
func (m *ContractManager) Reload(ctx context.Context) error {
	addrs, err := m.db.GetActiveInstances(ctx)
	if err != nil {
		return err
	}

	m.mu.Lock()
	defer m.mu.Unlock()

	// 转换 string -> common.Address
	m.contracts = make([]common.Address, len(addrs))
	for i, addr := range addrs {
		m.contracts[i] = common.HexToAddress(addr)
	}
	log.Printf("Loaded %d active contracts", len(m.contracts))
	return nil
}

// GetContracts 获取当前缓存的合约列表
func (m *ContractManager) GetContracts() []common.Address {
	m.mu.RLock()
	defer m.mu.RUnlock()
	// 返回副本以防并发问题
	result := make([]common.Address, len(m.contracts))
	copy(result, m.contracts)
	return result
}

// StartAutoReload 周期性刷新缓存（例如每分钟，或者由 FactoryListener 触发）
func (m *ContractManager) StartAutoReload(ctx context.Context, interval time.Duration) {
	ticker := time.NewTicker(interval)
	defer ticker.Stop()

	// 启动时先加载一次
	m.Reload(ctx)

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			if err := m.Reload(ctx); err != nil {
				log.Printf("Error reloading contracts: %v", err)
			}
		}
	}
}
