package main

import (
	"context"
	"log"
	"time"

	"chain-sql/internal/blockchain"
	"chain-sql/internal/config"
	"chain-sql/internal/core"
	"chain-sql/internal/database"
)

func main() {
	// 1. 加载配置
	cfg, err := config.Load("configs/config.yaml")
	if err != nil {
		log.Fatalf("Failed to load config: %v", err)
	}
	log.Printf("Config loaded. ChainID: %d, RPC: %s", cfg.Chain.ChainID, cfg.Chain.RPCURL)

	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	// 2. 连接数据库
	db, err := database.New(ctx, cfg.Database.DSN)
	if err != nil {
		log.Fatalf("Failed to connect to database: %v", err)
	}
	defer db.Close()
	log.Println("Database connected and schema initialized")

	// 3. 连接区块链节点
	ethClient, err := blockchain.NewClient(cfg.Chain.RPCURL)
	if err != nil {
		log.Fatalf("Failed to connect to blockchain: %v", err)
	}
	defer ethClient.Close()

	// 测试节点连接
	blockNum, err := ethClient.BlockNumber(ctx)
	if err != nil {
		log.Printf("Warning: Failed to get block number (is node running?): %v", err)
	} else {
		log.Printf("Connected to Ethereum node. Current Block: %d", blockNum)
	}

	factoryListener := core.NewFactoryListener(cfg, ethClient, db)
	// 使用 goroutine 启动，保持 main 不退出
	go factoryListener.Start(ctx)

	// 5. 初始化 Contract Manager
	mgr := core.NewContractManager(db)
	go mgr.StartAutoReload(ctx, 30*time.Second) // 每30秒刷新一次列表

	// 6. 启动 Data Listener
	dataListener := core.NewDataListener(cfg, ethClient, db, mgr)
	go dataListener.Start(ctx)

	// 阻塞主线程，等待信号退出
	<-ctx.Done()
	log.Println("Shutting down...")

	log.Println("System infrastructure initialization successful!")
}
