package core

import (
	"context"
	"fmt"
	"log"
	"math/big"
	"os"
	"strings"
	"time"

	"chain-sql/internal/blockchain"
	"chain-sql/internal/config"
	"chain-sql/internal/database"
	"chain-sql/internal/executor"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
)

const SyncKeyData = "data_listener"

// --- Event Data Structures ---

type EvTableCreated struct {
	TableName string
	Columns   []struct {
		Name         string
		SqlType      string
		IsPrimaryKey bool
	}
}

type EvDataInserted struct {
	TableName string
	Columns   []string
	Values    []string
}

// --- Listener Implementation ---

type DataListener struct {
	cfg    *config.Config
	client blockchain.Client
	db     *database.DB

	// 修改点：使用接口，不再依赖具体实现
	manager  ContractProvider
	executor EventHandler

	contractABI abi.ABI
}

func NewDataListener(cfg *config.Config, client blockchain.Client, db *database.DB, manager ContractProvider) *DataListener {
	// 动态加载 ABI
	abiPath := "configs/abi.json"
	abiBytes, err := os.ReadFile(abiPath)
	if err != nil {
		log.Fatalf("Failed to read ABI file %s: %v", abiPath, err)
	}

	parsedABI, err := abi.JSON(strings.NewReader(string(abiBytes)))
	if err != nil {
		log.Fatalf("Failed to parse ABI JSON: %v", err)
	}

	log.Printf("DataListener initialized with ABI containing %d events", len(parsedABI.Events))

	// 创建具体的 Executor 实现，但在 Struct 中作为 EventHandler 接口存储
	realExecutor := executor.NewExecutor(db)

	return &DataListener{
		cfg:         cfg,
		client:      client,
		db:          db,
		manager:     manager,
		executor:    realExecutor, // 注入实现
		contractABI: parsedABI,
	}
}

func (l *DataListener) Start(ctx context.Context) {
	log.Println("Starting Data Listener...")
	ticker := time.NewTicker(l.cfg.Sync.PollInterval)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			if err := l.syncBatch(ctx); err != nil {
				log.Printf("Error syncing data events: %v", err)
			}
		}
	}
}

func (l *DataListener) syncBatch(ctx context.Context) error {
	contracts := l.manager.GetContracts()
	if len(contracts) == 0 {
		return nil
	}

	// 1. 获取同步游标
	lastBlock, err := l.db.GetLastBlock(ctx, SyncKeyData)
	if err != nil {
		return fmt.Errorf("get cursor failed: %w", err)
	}
	if lastBlock == 0 {
		lastBlock = l.cfg.Sync.StartBlock
	}
	fromBlock := lastBlock + 1

	// 2. 获取链头
	headBlock, err := l.client.BlockNumber(ctx)
	if err != nil {
		return fmt.Errorf("get head block failed: %w", err)
	}

	// 3. 计算 ToBlock
	safeHead := headBlock - l.cfg.Sync.Confirmations
	if fromBlock > safeHead {
		return nil // 尚未达到确认高度
	}

	toBlock := fromBlock + uint64(l.cfg.Sync.MaxBatchSize)
	if toBlock > safeHead {
		toBlock = safeHead
	}

	// 4. 准备 Event Topics
	var topics []common.Hash
	for _, event := range l.contractABI.Events {
		topics = append(topics, event.ID)
	}

	// 5. 分批轮询地址
	batchSize := l.cfg.Sync.AddressBatchSize
	if batchSize <= 0 {
		batchSize = 50
	}

	log.Printf("Scanning Data: %d -> %d (%d contracts)", fromBlock, toBlock, len(contracts))

	for i := 0; i < len(contracts); i += batchSize {
		end := i + batchSize
		if end > len(contracts) {
			end = len(contracts)
		}
		batchAddrs := contracts[i:end]

		query := ethereum.FilterQuery{
			FromBlock: big.NewInt(int64(fromBlock)),
			ToBlock:   big.NewInt(int64(toBlock)),
			Addresses: batchAddrs,
			Topics:    [][]common.Hash{topics},
		}

		logs, err := l.client.FilterLogs(ctx, query)
		if err != nil {
			log.Printf("FilterLogs failed for batch %d: %v", i, err)
			continue
		}

		for _, vLog := range logs {
			if err := l.processLog(ctx, vLog); err != nil {
				log.Printf("Failed to process log tx=%s: %v", vLog.TxHash.Hex(), err)
			}
		}
	}

	// 6. 更新游标
	return l.db.UpdateLastBlock(ctx, SyncKeyData, toBlock)
}

func (l *DataListener) processLog(ctx context.Context, vLog types.Log) error {
	contractAddr := vLog.Address.Hex()

	event, err := l.contractABI.EventByID(vLog.Topics[0])
	if err != nil {
		return nil
	}

	switch event.Name {
	case "TableCreated":
		var ev EvTableCreated
		if err := l.contractABI.UnpackIntoInterface(&ev, event.Name, vLog.Data); err != nil {
			return fmt.Errorf("unpack TableCreated error: %w", err)
		}

		// 结构体转换以适配接口定义
		cols := make([]struct {
			Name         string
			SqlType      string
			IsPrimaryKey bool
		}, len(ev.Columns))
		for i, c := range ev.Columns {
			cols[i] = struct {
				Name         string
				SqlType      string
				IsPrimaryKey bool
			}{c.Name, c.SqlType, c.IsPrimaryKey}
		}

		log.Printf("[DDL] TableCreated: %s.%s", contractAddr, ev.TableName)
		return l.executor.HandleTableCreated(ctx, contractAddr, ev.TableName, cols)

	case "DataInserted":
		var ev EvDataInserted
		if err := l.contractABI.UnpackIntoInterface(&ev, event.Name, vLog.Data); err != nil {
			return fmt.Errorf("unpack DataInserted error: %w", err)
		}

		log.Printf("[DML] DataInserted: %s.%s", contractAddr, ev.TableName)
		return l.executor.HandleDataInserted(ctx, contractAddr, ev.TableName, ev.Columns, ev.Values)

	default:
		// 其他事件忽略
	}

	return nil
}
