// SPDX-License-Identifier: MIT
pragma solidity ^0.8.0;

import "./ISQLSync.sol";

contract SQLSync is ISQLSync {
    address public owner;

    modifier onlyOwner() {
        require(msg.sender == owner, "Not owner");
        _;
    }

    // 关键修改：接收 _owner 参数
    constructor(address _owner) {
        owner = _owner;
    }

    // --- DDL ---
    function createTable(
        string calldata tableName,
        ColumnDef[] calldata columns
    ) external override onlyOwner {
        emit TableCreated(tableName, columns);
    }

    function dropTable(string calldata tableName) external override onlyOwner {
        emit TableDropped(tableName);
    }

    function alterTable(
        string calldata tableName,
        string calldata rawSql
    ) external override onlyOwner {
        emit TableAltered(tableName, rawSql);
    }

    function createIndex(
        string calldata tableName,
        string calldata indexName,
        string[] calldata columns,
        bool isUnique
    ) external override onlyOwner {
        emit IndexCreated(tableName, indexName, columns, isUnique);
    }

    function truncateTable(
        string calldata tableName
    ) external override onlyOwner {
        emit TableTruncated(tableName);
    }

    // --- DML ---
    function insert(
        string calldata tableName,
        string[] calldata columns,
        string[] calldata values
    ) external override onlyOwner {
        emit DataInserted(tableName, columns, values);
    }

    function update(
        string calldata tableName,
        string[] calldata setColumns,
        string[] calldata setValues,
        string calldata whereClause
    ) external override onlyOwner {
        emit DataUpdated(tableName, setColumns, setValues, whereClause);
    }

    function deleteData(
        string calldata tableName,
        string calldata whereClause
    ) external override onlyOwner {
        emit DataDeleted(tableName, whereClause);
    }

    function batchInsert(
        string calldata tableName,
        string[] calldata columns,
        string[] calldata flatValues
    ) external override onlyOwner {
        // 这里只是为了演示接口，实际业务逻辑可以加参数校验
        emit DataBatchInserted(
            tableName,
            columns,
            _reconstructBatch(flatValues, columns.length)
        );
    }

    function upsert(
        string calldata tableName,
        string[] calldata columns,
        string[] calldata values,
        string calldata conflictColumn
    ) external override onlyOwner {
        emit DataUpserted(tableName, columns, values, conflictColumn);
    }

    // 辅助函数：将一维数组重构为二维数组以匹配 Event 定义 (Gas 消耗较高，生产环境建议直接传 bytes)
    function _reconstructBatch(
        string[] memory flatValues,
        uint256 colCount
    ) internal pure returns (string[][] memory) {
        require(colCount > 0, "Col count 0");
        require(
            flatValues.length % colCount == 0,
            "Flat values length mismatch"
        );

        uint256 rowCount = flatValues.length / colCount;
        string[][] memory result = new string[][](rowCount);

        for (uint256 i = 0; i < rowCount; i++) {
            string[] memory row = new string[](colCount);
            for (uint256 j = 0; j < colCount; j++) {
                row[j] = flatValues[i * colCount + j];
            }
            result[i] = row;
        }
        return result;
    }
}
