package core

import (
	"context"
	"fmt"
	"log"
	"math/big"
	"time"

	"chain-sql/internal/blockchain"
	"chain-sql/internal/config"
	"chain-sql/internal/database"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
)

const (
	SyncKeyFactory = "factory_listener"
)

type FactoryListener struct {
	cfg         *config.Config
	client      blockchain.Client
	db          *database.DB
	factoryAddr common.Address
	// 缓存 Event 签名
	eventSig common.Hash
	eventABI abi.Event
}

func NewFactoryListener(cfg *config.Config, client blockchain.Client, db *database.DB) *FactoryListener {
	// 手动定义 ABI，或者使用 abigen 生成的代码。这里为了演示独立性手动定义。
	// Event: InstanceCreated(address indexed owner, address indexed instance)
	eventSig := crypto.Keccak256Hash([]byte("InstanceCreated(address,address)"))

	return &FactoryListener{
		cfg:         cfg,
		client:      client,
		db:          db,
		factoryAddr: common.HexToAddress(cfg.Chain.FactoryAddress),
		eventSig:    eventSig,
	}
}

func (l *FactoryListener) Start(ctx context.Context) {
	log.Println("Starting Factory Listener...")
	ticker := time.NewTicker(l.cfg.Sync.PollInterval)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
			if err := l.syncOnce(ctx); err != nil {
				log.Printf("Error syncing factory events: %v", err)
			}
		}
	}
}

func (l *FactoryListener) syncOnce(ctx context.Context) error {
	// 1. 获取上次同步高度
	lastBlock, err := l.db.GetLastBlock(ctx, SyncKeyFactory)
	if err != nil {
		return fmt.Errorf("get cursor failed: %w", err)
	}

	// 2. 如果是从 0 开始，使用配置的 StartBlock
	if lastBlock == 0 {
		lastBlock = l.cfg.Sync.StartBlock
	}

	fromBlock := lastBlock + 1

	// 3. 获取当前链高度
	headBlock, err := l.client.BlockNumber(ctx)
	if err != nil {
		return fmt.Errorf("get head block failed: %w", err)
	}

	// 4. 计算 ToBlock (考虑确认数和BatchSize)
	safeHead := headBlock - l.cfg.Sync.Confirmations
	if fromBlock > safeHead {
		return nil // 还没到安全高度
	}

	toBlock := fromBlock + uint64(l.cfg.Sync.MaxBatchSize)
	if toBlock > safeHead {
		toBlock = safeHead
	}

	log.Printf("Scanning Factory: %d -> %d", fromBlock, toBlock)

	// 5. 拉取日志
	query := ethereum.FilterQuery{
		FromBlock: big.NewInt(int64(fromBlock)),
		ToBlock:   big.NewInt(int64(toBlock)),
		Addresses: []common.Address{l.factoryAddr},
		Topics:    [][]common.Hash{{l.eventSig}},
	}

	logs, err := l.client.FilterLogs(ctx, query)
	if err != nil {
		return fmt.Errorf("filter logs failed: %w", err)
	}

	// 6. 处理日志
	for _, vLog := range logs {
		if len(vLog.Topics) < 3 {
			continue // 异常日志
		}
		// topic[0] is signature, topic[1] is owner (indexed), topic[2] is instance (indexed)
		owner := common.HexToAddress(vLog.Topics[1].Hex())
		instance := common.HexToAddress(vLog.Topics[2].Hex())

		log.Printf("New Instance Found! Address: %s, Owner: %s", instance.Hex(), owner.Hex())

		// 存入 DB
		if err := l.db.RegisterInstance(ctx, instance.Hex(), owner.Hex(), vLog.BlockNumber); err != nil {
			log.Printf("Failed to register instance %s: %v", instance.Hex(), err)
			// 这里策略：如果是 DB 错误，应该 retry 或 return err 阻止 cursor 更新
			// 简单起见，这里 return err
			return err
		}
	}

	// 7. 更新游标
	if err := l.db.UpdateLastBlock(ctx, SyncKeyFactory, toBlock); err != nil {
		return fmt.Errorf("update cursor failed: %w", err)
	}

	return nil
}
