#!/bin/bash
# scripts/docker-deploy.sh
# Docker 部署脚本

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

echo -e "${GREEN}🚀 Deploying ChainSQL with Docker Compose${NC}"

# 检查 .env 文件
if [ ! -f .env ]; then
    echo -e "${YELLOW}⚠️  .env file not found, creating from .env.example${NC}"
    cp .env.example .env
    echo -e "${RED}❗ Please edit .env file with your configuration before deploying!${NC}"
    exit 1
fi

# 检查必要的环境变量
source .env

if [ "$CHAIN_FACTORY_ADDRESS" = "0x..." ]; then
    echo -e "${RED}❌ Please set CHAIN_FACTORY_ADDRESS in .env file${NC}"
    exit 1
fi

# 创建日志目录
mkdir -p logs

# 停止旧服务（如果存在）
echo -e "${YELLOW}Stopping existing services...${NC}"
docker-compose down 2>/dev/null || true

# 构建镜像
echo -e "${YELLOW}Building images...${NC}"
docker-compose build

# 启动服务
echo -e "${YELLOW}Starting services...${NC}"
docker-compose up -d

# 等待服务启动
echo -e "${YELLOW}Waiting for services to be healthy...${NC}"
sleep 5

# 检查服务状态
echo -e "\n${GREEN}Service Status:${NC}"
docker-compose ps

# 检查健康状态
echo -e "\n${YELLOW}Checking health status...${NC}"
for i in {1..30}; do
    if curl -sf http://localhost:8080/health > /dev/null 2>&1; then
        echo -e "${GREEN}✅ ChainSQL is healthy!${NC}"
        break
    fi
    echo -n "."
    sleep 1
done

echo -e "\n${GREEN}🎉 Deployment completed!${NC}"
echo -e "${YELLOW}Useful commands:${NC}"
echo -e "  View logs:    docker-compose logs -f chainsql"
echo -e "  Stop:         docker-compose down"
echo -e "  Restart:      docker-compose restart chainsql"
echo -e "  Health check: curl http://localhost:8080/health"
