package opencl

import (
	"bytes"
	"encoding/hex"
	"fmt"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/xecc/types"
	"io/ioutil"
	"testing"
	"time"
)

type recoverCase struct {
	hash string
	rsig string
	pubk string
}

var recoverArray = []recoverCase{
	{
		hash: "31920d1ef030c8296c85c29e4658b1a29e2fea41af7ddad24dc6bc59c6e23c8d",
		rsig: "ccc8e751834ddacc76773c9638f506e049840166eb025152ec5fb7d34b50feb06d95e1f220c8031ed73e8608f09f5de14cddb8122c3c0ec4ce26c42a2274e80b01",
		pubk: "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3f",
	},
	{
		hash: "e5312bc72030bb1c356e29f6047bd663e663d096ada0beb858081a4d3b55bf12",
		rsig: "290a4d4e585f28db796dae3ced74130fabffead3b5c2923304e1761fb9ca936a09728351c56a4294df0c1fc387931d5cff7f59eadb1084bace16af61a341ef8601",
		pubk: "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3f",
	},
	{
		hash: "e04ab1e1892445ed18c7caa515e593e69eddea7250a9076bffbbc6aac87593ae",
		rsig: "f8e53a45b4607cc4044439b4caac56179df941543f8bab6c6be6de681f7e1ee467591d058ab8660cf103b89ea749e75e2cd3c2b34fac4b3620b1b37b0542a72500",
		pubk: "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3f",
	},
	{
		hash: "72804c483b95fb866712ca175bd431214f46916fad5437b577772b0dd135e9f3",
		rsig: "8fe82a947d922451f2fe2011bb63311248ff79fb9a88d2d28d3779ae80b7090664d9990c1f3077ca4f691af9594974f0be979a3801f65705757b86344fbe2b6601",
		pubk: "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3f",
	},
	{
		hash: "0824caafe5252483719d49ce28e0968d5f927d24ce96b31c5de48d0133ab3373",
		rsig: "507f3caa9b0a3ccbf0e063fa21f0e765dab423a667268d9ce7101d2b8034e6fc44cb391b950fe8306e07b46e8df8bc9e10dd12c44100382861420d7f315db80f00",
		pubk: "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3f",
	},
}

func init() {
	e := GetInstance()
	for !e.Ready() {
		fmt.Println("wait opencl init")
		time.Sleep(time.Second)
	}
}

func TestGetCLCode(t *testing.T) {
	origin, err := ioutil.ReadFile("code/k.cl")
	if err != nil {
		t.Error("read origin data failed")
	}
	fsk, err := getCLSourcode()
	if err != nil {
		t.Error("get fsk data failed")
	}
	if bytes.Compare(origin, fsk) != 0 {
		ioutil.WriteFile("fsk.cl", fsk, 0755)
		t.Error("content compare failed")
	}
}

func TestRecover(t *testing.T) {
	ocl := GetInstance()
	strhash := "e5312bc72030bb1c356e29f6047bd663e663d096ada0beb858081a4d3b55bf12"
	strsig := "290a4d4e585f28db796dae3ced74130fabffead3b5c2923304e1761fb9ca936a09728351c56a4294df0c1fc387931d5cff7f59eadb1084bace16af61a341ef8601"
	strpub := "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3f"
	//error pub
	//strpub := "0481903830e01769fe48c372c1c02ce7fefbf042c3fde1f1cb74d3ada4465e26eeacb1f020b347614ab7e2dedb71e2efaa89e2a8661960839f72f6dde95d5eff3s"

	var msg, sig []byte
	msg, _ = hex.DecodeString(strhash)
	sig, _ = hex.DecodeString(strsig)
	expectpub, _ := hex.DecodeString(strpub)

	ts := time.Now().Nanosecond()
	pub, err := ocl.OclSecp256RecoverPubkeyS(msg, sig)
	if err != nil {
		t.Error("Recover failed")
	}
	te := time.Now().Nanosecond()
	if bytes.Compare(pub, expectpub) != 0 {
		t.Error("compare failed")
		t.Log("pub", hex.EncodeToString(pub))
	}
	fmt.Printf("recover cost %dus\n", (te-ts)/1000)
}

func TestVerify(t *testing.T) {
	strhash := "02030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f2021"
	strsig := "e8a97e31f8f5dde21b7394b190ed149ba0e28ddfdd44a170d21eb60f37bd132c04909be5273908befbe6166a9cb3f38fbb19c8cfff1d1561f4d61dfa0c499a81"
	strpub := "041d17bd8d96b4bdf49dd1436e5af4f0bd3bff939df8cf8fdc1f621e3c89de2a47111eb88b4bb89f180702a8bc744d5a37615965713d1fdfff000c2a62b4aea409"

	// error pub
	//strpub := "041d17bd8d96b4bdf49dd1436e5af4f0bd3bff939df8cf8fdc1f621e3c89de2a47111eb88b4bb89f180702a8bc744d5a37615965713d1fdfff000c2a62b4aea404"

	var msg, sig []byte
	msg, _ = hex.DecodeString(strhash)
	sig, _ = hex.DecodeString(strsig)
	pub, _ := hex.DecodeString(strpub)
	ts := time.Now().Nanosecond()
	verify, err := GetInstance().OclSecp256VerifyS(msg, sig, pub)
	if err != nil || !verify {
		t.Error("Verify failed")
	} else {
		t.Log("Verify successful")
	}
	te := time.Now().Nanosecond()
	fmt.Printf("verify cost %dus\n", (te-ts)/1000)
}

func TestProcess(t *testing.T) {
	var tasks = make([]types.XTask, 0)
	var okpubs = make([][]byte, 0)
	for _, info := range recoverArray {
		task := &types.XTaskSecp256k1RPubkey{}
		task.Msg, _ = hex.DecodeString(info.hash)
		task.Rsig, _ = hex.DecodeString(info.rsig)
		pub, _ := hex.DecodeString(info.pubk)
		tasks = append(tasks, task)
		okpubs = append(okpubs, pub)
	}

	for i, tm := range tasks {
		fmt.Println("test index ", i)
		ts, e := GetInstance().Process(tm)
		if e != nil {
			t.Error("process failed", e)
		}
		if bytes.Compare(okpubs[i], tm.(*types.XTaskSecp256k1RPubkey).Pubkey) != 0 {
			fmt.Println("compare pubkey failed", "got", hex.EncodeToString(tm.(*types.XTaskSecp256k1RPubkey).Pubkey), "expect ", hex.EncodeToString(okpubs[i]))
		} else {
			fmt.Println("tm got correct pubkey")
		}

		if bytes.Compare(okpubs[i], ts.(*types.XTaskSecp256k1RPubkey).Pubkey) != 0 {
			fmt.Println("compare pubkey failed", "ret got", hex.EncodeToString(ts.(*types.XTaskSecp256k1RPubkey).Pubkey), "expect ", hex.EncodeToString(okpubs[i]))
		} else {
			fmt.Println("return value got correct pubkey")
		}
	}
}

func TestProcessBatch(t *testing.T) {
	var tasks = make([]types.XTask, 0)
	var okpubs = make([][]byte, 0)
	for _, info := range recoverArray {
		task := &types.XTaskSecp256k1RPubkey{}
		task.Msg, _ = hex.DecodeString(info.hash)
		task.Rsig, _ = hex.DecodeString(info.rsig)
		pub, _ := hex.DecodeString(info.pubk)
		tasks = append(tasks, task)
		okpubs = append(okpubs, pub)
	}

	rets, e := GetInstance().ProcessBatch(tasks)
	if e != nil {
		t.Error("process failed", e)
	}
	for i, m := range rets {
		if bytes.Compare(okpubs[i], m.(*types.XTaskSecp256k1RPubkey).Pubkey) != 0 {
			t.Error("compare pubkey failed", "got", hex.EncodeToString(m.(*types.XTaskSecp256k1RPubkey).Pubkey), "expect ", hex.EncodeToString(okpubs[i]))
		} else {
			fmt.Println("return value ok， task", i)
		}
	}

	for i, m := range tasks {
		if bytes.Compare(okpubs[i], m.(*types.XTaskSecp256k1RPubkey).Pubkey) != 0 {
			t.Error("compare pubkey failed", "got", hex.EncodeToString(m.(*types.XTaskSecp256k1RPubkey).Pubkey), "expect ", hex.EncodeToString(okpubs[i]))
		} else {
			fmt.Println("param value ok, task", i)
		}
	}

}
