package main

import (
	"fmt"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/common/log"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/config"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/service"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/worker"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/xecc"
	"github.com/fsnotify/fsnotify"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
	"google.golang.org/grpc"
	"net"
	"os"
	"path"
	"path/filepath"
	"sync"
)

var cfgFile string
var Verbose bool
var routineCount uint

// RootCmd represents the base command when called without any subcommands
var RootCmd = &cobra.Command{
	Use:   "metacryptor",
	Short: "The MetaCryptor command-line interface",
	Long:  ``,
	Run: func(cmd *cobra.Command, args []string) {
		runNode()
	},
	// Uncomment the following line if your bare application
	// has an action associated with it:
	//	Run: func(cmd *cobra.Command, args []string) { },
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		log.Fatal(err)
		os.Exit(-1)
	}
}

func init() {
	cobra.OnInitialize(initConfig)

	RootCmd.PersistentFlags().UintVar(&routineCount, "routine", 2, "routine count for corrupt do task")
	RootCmd.PersistentFlags().StringVar(&cfgFile, "config", "config.toml", "config file (default is ./config.yaml)")
	RootCmd.PersistentFlags().BoolVarP(&Verbose, "verbose", "v", false, "verbose output")
}

// initConfig reads in config file and ENV variables if set.
func initConfig() {
	log.InitLog()

	dir, err := filepath.Abs(filepath.Dir(os.Args[0]))
	if err != nil {
		log.Fatal(err)
	}

	viper.SetConfigName("config") // name of config file (without extension)
	if cfgFile != "" {            // enable ability to specify config file via flag
		//log.Warn(">>> cfgFile: ", cfgFile)
		viper.SetConfigFile(cfgFile)
		configDir := path.Dir(cfgFile)
		if configDir != "." && configDir != dir {
			viper.AddConfigPath(configDir)
		}
	}

	viper.AddConfigPath(".")
	viper.AutomaticEnv() // read in environment variables that match

	// If a config file is found, read it in.
	if err := viper.ReadInConfig(); err == nil {
		log.Info("Using config file:", viper.ConfigFileUsed())
	} else {
		log.Error("Read config failed", "error", err)
		return
	}

	_, err = config.ParseConfig(viper.ConfigFileUsed())
	if err != nil {
		log.WithField("error", err).Fatal("parse config failed")
		return
	}
	//log.Info("config is", config.GetConfig())
	config.GetConfig().SetRoutineCount(routineCount)

	viper.WatchConfig()
	viper.OnConfigChange(func(e fsnotify.Event) {
		log.Warn("Config file changed:", e.Name)
	})
}

func runNode() {
	xecc.XeccInstance()
	wk := worker.NewWorker()
	wk.Start()
	lis, err := net.Listen("tcp", config.GetConfig().GpcAddress)
	if err != nil {
		fmt.Printf("failed to listen: %v", err)
		return
	}
	s := grpc.NewServer()
	service.RegisterCrypter(s, wk)

	err = s.Serve(lis)
	if err != nil {
		fmt.Printf("failed to serve: %v", err)
		return
	}

	wg := sync.WaitGroup{}
	wg.Add(1)

	wg.Wait()
}
