package eg

import (
	"ChainGrpcTest/log"
	"ChainGrpcTest/types"
	"context"
	"encoding/hex"
	"flag"
	basetype "github.com/CaduceusMetaverseProtocol/MetaProtocol/gen/proto/go/base/v1"
	metanebula "github.com/CaduceusMetaverseProtocol/MetaProtocol/gen/proto/go/nebula/v1"
	metatypes "github.com/CaduceusMetaverseProtocol/MetaTypes/types"
	gogotypes "github.com/gogo/protobuf/types"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"math/big"
	"strings"
)

func dumpHash(h *metatypes.Hash) string {
	if h != nil {
		return h.String()
	} else {
		return "nil"
	}
}

func dumpBigInt(n *metatypes.BigInt) string {
	if n == nil {
		return "nil"
	}
	return types.ToBigInt(n).String()
}

func dumpBloom(b *basetype.Bloom) string {
	if b == nil {
		return "nil"
	}
	return hex.EncodeToString(b.Data)
}

func dumpLogs(txlogs []*basetype.MetaTxLog) string {
	if txlogs == nil || len(txlogs) == 0 {
		return "empty"
	}
	return "not empty"
}

func dumpAddress(a *metatypes.Address) string {
	if a == nil {
		return "nil"
	}
	return a.String()
}

func PrintReceipt(receipt *basetype.MetaReceipt) {
	log.Info("receipt.type=", receipt.Type)
	log.Info("receipt.txHash=", dumpHash(receipt.TxHash))
	log.Info("receipt.blockNumber=", dumpBigInt(receipt.BlockNumber))
	log.Info("receipt.blockHash=", dumpHash(receipt.BlockHash))
	log.Info("receipt.Status=", receipt.Status)
	log.Info("receipt.TxIndex=", receipt.TxIndex)
	log.Info("receipt.ContractAddress=", dumpAddress(receipt.ContractAddress))
	log.Info("receipt.Log=", dumpLogs(receipt.Logs))
	log.Info("receipt.Root=", hex.EncodeToString(receipt.Root))
	log.Info("receipt.CumulativeGasUsed=", receipt.CumulativeGasUsed)
	log.Info("receipt.GasUsed=", receipt.GasUsed)
}

func PrintProofTx(tx *basetype.MetaProofTx) {
	log.Info("tx.from=", dumpAddress(tx.Base.From))
	log.Info("tx.to=", dumpAddress(tx.Base.To))
	log.Info("tx.value=", dumpBigInt(tx.Base.Value))
	log.Info("tx.hash=", dumpHash(tx.Base.TxHash))
	log.Info("tx.type=", tx.Base.TxType)
	log.Info("tx.nonce=", tx.Base.Nonce)
	log.Info("tx.gasPrice=", dumpBigInt(tx.Base.GasPrice))
	log.Info("tx.gasLimit=", tx.Base.Gas)
	log.Info("tx.chainId=", dumpBigInt(tx.Base.ChainId))
	log.Info("tx.expireBlock=", dumpBigInt(tx.Base.ExpiredBlock))
	//log.Info("tx.receiveTime=", tx.Base.ReceiveTime)
	log.Info("tx.data=", hex.EncodeToString(tx.Base.Data))
	log.Info("tx.V=", dumpBigInt(tx.Base.V))
	log.Info("tx.R=", dumpBigInt(tx.Base.R))
	log.Info("tx.S=", dumpBigInt(tx.Base.S))
}

func PrintTransaction(transaction *basetype.MetaTransaction) {
	prooftx := new(basetype.MetaProofTx)
	if gogotypes.Is(transaction.Tx, prooftx) {
		gogotypes.UnmarshalAny(transaction.Tx, prooftx)
	} else {
		log.Errorf("unknown block transaction type:%T\n", transaction.Tx)
	}
	PrintProofTx(prooftx)

}
func PrintBlock(block *basetype.MetaBlock) {
	if block == nil {
		log.Info("block is nil")
		return
	}
	if block.Header == nil {
		log.Info("block.header is nil")
	} else {
		h := block.GetHeader()
		log.Info("block.BlockHash=", dumpHash(h.BlockHash))
		log.Info("block.BlockNumber=", types.ToBigInt(block.Header.BlockNumber))
		log.Info("block.TxsRoot=", dumpHash(h.TxsRoot))
		log.Info("block.Timestamp=", h.Timestamp)
		log.Info("block.ReceiptsRoot=", dumpHash(h.ReceiptsRoot))
		log.Info("block.ParentHash=", dumpHash(h.ParentHash))
		log.Info("block.Extra=", hex.EncodeToString(h.Extra))
		log.Info("block.StateRoot=", dumpHash(h.StateRoot))
		log.Info("block.BlockBloom=", dumpBloom(h.BlockBloom))
		log.Info("block.GasLimit=", h.GasLimit)
		log.Info("block.GasUsed=", h.GasUsed)
		log.Info("block.Miner=", h.Miner)
	}
	if block.Txs == nil {
		log.Info("block.txs is nil")
	} else {
		log.Info("block.txs count=", len(block.Txs))
		for i, tx := range block.Txs {
			log.Infof("block.txs[%d]:", i)
			PrintTransaction(tx)
		}
	}
}

func featureBlock(client metanebula.NebulaServiceClient, number int64, hash string) {
	if len(hash) > 0 {
		req := new(metanebula.BlockByHashRequest)
		req.BlockHash = (*metatypes.Hash)(types.HexToHash(hash).Bytes())
		res, err := client.BlockByHash(context.Background(), req)
		if err != nil {
			log.Errorf("block by hash failed with err=%s", err)
			return
		}
		PrintBlock(res.Block)
	} else {
		req := new(metanebula.BlockByNumberRequest)
		req.BlockId = types.FromBigInt(big.NewInt(number))
		res, err := client.BlockByNumber(context.Background(), req)
		if err != nil {
			log.Errorf("block by number failed with err=%s", err)
			return
		}
		PrintBlock(res.Block)
	}
}

func featureReceipt(client metanebula.NebulaServiceClient, hash string) {
	req := new(metanebula.TransactionReceiptRequest)
	req.TxHash = (*metatypes.Hash)(types.HexToHash(hash).Bytes())
	res, err := client.TransactionReceipt(context.Background(), req)
	if err != nil {
		log.Errorf("receipt by hash failed with err=%s", err)
		return
	}
	PrintReceipt(res.TxReceipt)
}

func featureTransaction(client metanebula.NebulaServiceClient, hash string) {
	req := new(metanebula.TransactionByHashRequest)
	req.TxHash = (*metatypes.Hash)(types.HexToHash(hash).Bytes())
	res, err := client.TransactionByHash(context.Background(), req)
	if err != nil {
		log.Errorf("receipt by hash failed with err=%s", err)
		return
	}
	PrintTransaction(res.TxData)
}

func featureTestSendTransaction(client metanebula.NebulaServiceClient) {
	prooftx := &basetype.MetaProofTx{
		Base: &basetype.MetaTxBase{
			TxHash: (*metatypes.Hash)(types.HexToHash("0x7b66e1b27b55febba2407ee31e76245d3a0d4254dafbdaa4dcde36d9efd15226").Bytes()),
			TxType: 2,
			Value:  metatypes.NewBigInt(122222),
		},
	}
	anybase, err := gogotypes.MarshalAny(prooftx)
	if err != nil {
		log.Errorf("marshal any failed with err:%s", err)
		return
	}

	mtx := &basetype.MetaTransaction{
		Tx: anybase,
	}

	req := new(metanebula.TestSendTransactionRequest)
	req.TxData = mtx
	res, err := client.TestSendTransaction(context.Background(), req)
	if err != nil {
		log.Errorf("TestSendTransaction with err=%s", err)
		return
	}
	log.Infof("TestSendTransaction succeed got res.TxHash:%s", res.TxHash)

}

func featureAccountInfo(client metanebula.NebulaServiceClient, address string) {
	addr := metatypes.HexToAddress(address)
	getbalance := new(metanebula.BalanceRequest)
	getbalance.Address = &addr
	res, err := client.Balance(context.Background(), getbalance)
	if err != nil {
		log.Errorf("get balance failed with err=%s", err)
		return
	}
	log.Infof("get account(%s).balance=%s", addr.String(), res.Balance.Text(16))

	getnonce := new(metanebula.NonceRequest)
	getnonce.Address = &addr
	nonceres, err := client.Nonce(context.Background(), getnonce)
	if err != nil {
		log.Errorf("get nonce failed with err=%s", err)
		return
	}
	log.Infof("get account(%s).nonce=%v", addr.String(), nonceres.Nonce)
}

func featureNonceRepeated(client metanebula.NebulaServiceClient, addressList []string) {
	addrs := make([]metatypes.Address, len(addressList))
	for i, addr := range addressList {
		addrs[i] = metatypes.HexToAddress(addr)
	}
	getnonces := new(metanebula.RepeatedNonceRequest)
	getnonces.Address = addrs
	nonceres, err := client.RepeatedNonce(context.Background(), getnonces)
	if err != nil {
		log.Errorf("get nonce failed with err=%s", err)
		return
	}
	for i := 0; i < len(nonceres.Address); i++ {
		log.Infof("get account(%s).nonce=%v", nonceres.Address[i].String(), nonceres.Nonce[i])
	}
}

func main() {
	node := flag.String("node", "127.0.0.1:38004", "nebula service address")
	feature := flag.String("f", "block", "feature name, available value(block, receipt, tx, account)")
	number := flag.Int64("n", 10, "block number")
	hash := flag.String("hash", "", "request hash value, block/tx/receipt hash")
	acc := flag.String("addr", "", "get account info")
	flag.Parse()
	client, err := grpc.Dial(*node, grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		log.Error("dial nebula failed", "err", err)
		return
	}
	nebula := metanebula.NewNebulaServiceClient(client)
	switch *feature {
	case "block":
		featureBlock(nebula, *number, *hash)
	case "receipt":
		featureReceipt(nebula, *hash)
	case "send":
		featureTestSendTransaction(nebula)
	case "tx":
		featureTransaction(nebula, *hash)
	case "account":
		featureAccountInfo(nebula, *acc)
	case "repeatnonce":
		addresslist := strings.Split(*acc, ",")
		featureNonceRepeated(nebula, addresslist)
	}

	return
}
