package benchmark

import (
	"crypto/ecdsa"
	"math/big"
	"sync"

	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
)

func buildSendTx(nonce uint64, to common.Address, amount *big.Int, gasLimit uint64, gasPrice *big.Int, data []byte, chainId *big.Int, privateKey *ecdsa.PrivateKey) (*types.Transaction, error) {
	return buildTx(nonce, to, amount, gasLimit, gasPrice, data, chainId, privateKey)
}

func buildOriginalTx(nonce uint64, to common.Address, amount int64, chainId *big.Int, privateKey *ecdsa.PrivateKey) (*types.Transaction, error) {

	return buildTx(nonce, to, big.NewInt(amount), 0, big.NewInt(0), nil, chainId, privateKey)

}

func buildTx(nonce uint64, to common.Address, amount *big.Int, gasLimit uint64, gasPrice *big.Int, data []byte, chainId *big.Int, privateKey *ecdsa.PrivateKey) (*types.Transaction, error) {

	tx := types.NewTransaction(nonce, to, amount, gasLimit, gasPrice, data)
	if privateKey != nil {
		signedTx, err := types.SignTx(tx, types.NewEIP155Signer(chainId), privateKey)
		if err != nil {
			return nil, err
		}
		tx = signedTx
	}

	return tx, nil
}

type EthClient struct {
	PrivateKey *ecdsa.PrivateKey
	FromAddr   common.Address
	NodeUrl    string
	Nonce      uint64
	GasPrice   *big.Int
	ChainId    *big.Int
	GasLimit   uint64
}

var originalTxPrivateKey string = "9e0944f587e1043d6e303644738b0c7c77ed15b176ca574ed0be40c0b9bbdc3a"
var toAddress common.Address = common.HexToAddress("0x0071B39fd266F8aeF392fb50F078A233b2218a0b")
var originalTxsQueue chan *types.Transaction //= make(chan *types.Transaction, 50000)

var originalTxParam EthClient

var FromAddr common.Address

// type TxWithFrom struct {
// 	From []byte
// 	Tx   []byte
// }

func init() {

	originalTxPrivatekeyAsECDSA, err := crypto.HexToECDSA(originalTxPrivateKey)
	if err != nil {
		panic(err)
	}
	originalTxPublicKey := originalTxPrivatekeyAsECDSA.Public()
	originalTxPublicKeyECDSA, ok := originalTxPublicKey.(*ecdsa.PublicKey)
	if !ok {
		panic("publicKey.(*ecdsa.PublicKey) not ok")
	}
	originalTxFromAddress := crypto.PubkeyToAddress(*originalTxPublicKeyECDSA)

	originalTxParam = EthClient{
		PrivateKey: originalTxPrivatekeyAsECDSA,
		FromAddr:   originalTxFromAddress,
		Nonce:      0,
	}

	FromAddr = originalTxFromAddress

}

var doneOnce sync.Once

func ProduceOriginalTx(firstdone chan bool) error {

	if originalTxsQueue == nil {
		originalTxsQueue = make(chan *types.Transaction, 50000)
	}

	for {

		if len(originalTxsQueue) == 50000 {

			doneOnce.Do(func() {

				firstdone <- true
			})
		}

		tx, err := buildOriginalTx(originalTxParam.Nonce, toAddress, 0, big.NewInt(256), nil)
		if err != nil {
			return err
		}

		originalTxParam.Nonce += 1
		originalTxsQueue <- tx
	}
}

func ProduceOriginalTxByCount(count int64) (chan *types.Transaction, error) {

	var txsQueue chan *types.Transaction = make(chan *types.Transaction, count)

	for i := 0; int64(i) < count; i++ {

		tx, err := buildOriginalTx(originalTxParam.Nonce, toAddress, 0, big.NewInt(256), nil)
		if err != nil {
			return nil, err
		}

		originalTxParam.Nonce += 1
		txsQueue <- tx
	}

	return txsQueue, nil
}
