package crypto

import (
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/sha256"
	"errors"
	"fmt"
	"github.com/btcsuite/btcd/btcec"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/tjfoc/gmsm/sm2"
	"math/big"
)

//
func RecoverPubkey(msg, signature []byte) ([]byte, error) {
	return crypto.Ecrecover(msg, signature)
}

func RecoverableSign(msg, seckey []byte) ([]byte, error) {
	if len(msg) != 32 {
		return nil, fmt.Errorf("hash is required to be exactly 32 bytes (%d)", len(msg))
	}
	prv, _ := btcec.PrivKeyFromBytes(btcec.S256(), seckey)
	sig, err := btcec.SignCompact(btcec.S256(), (*btcec.PrivateKey)(prv), msg, false)
	if err != nil {
		return nil, err
	}
	// Convert to Ethereum signature format with 'recovery id' v at the end.
	v := sig[0] - 27
	copy(sig, sig[1:])
	sig[64] = v
	return sig, nil
}

func Secp256k1Sign(msg, seckey []byte) ([]byte, error) {
	priv, _ := btcec.PrivKeyFromBytes(btcec.S256(), seckey)
	msghash := sha256.Sum256(msg)
	sig, err := priv.Sign(msghash[:])
	if err != nil {
		return nil, err
	}
	rBytes := sig.R.Bytes()
	sBytes := sig.S.Bytes()
	sigBytes := make([]byte, 64)
	// 0 pad the byte arrays from the left if they aren't big enough.
	copy(sigBytes[32-len(rBytes):32], rBytes)
	copy(sigBytes[64-len(sBytes):64], sBytes)
	return sigBytes, nil
}

// warning: pubkey size is 64, not 65.
func Secp256k1Verify(msg, pubkey, sign []byte) (bool, error) {
	var secp256k1halfN = new(big.Int).Rsh(btcec.S256().N, 1)
	sig := &btcec.Signature{R: new(big.Int).SetBytes(sign[:32]), S: new(big.Int).SetBytes(sign[32:])}
	key, err := btcec.ParsePubKey(pubkey, btcec.S256())
	if err != nil {
		return false, errors.New(fmt.Sprintf("parse to secp256k1 pubkey invalid pubkey:%s", err.Error()))
	}
	// Reject malleable signatures. libsecp256k1 does this check but btcec doesn't.
	if sig.S.Cmp(secp256k1halfN) > 0 {
		return false, nil
	}
	return sig.Verify(msg, key), nil
}

func parseToEcdsaP256PrivateKey(seckey []byte) (*ecdsa.PrivateKey, error) {
	if len(seckey) != 32 {
		return nil, errors.New("invalid seckey")
	}
	k := big.NewInt(0).SetBytes(seckey)
	p256 := elliptic.P256()
	priv := new(ecdsa.PrivateKey)
	priv.PublicKey.Curve = p256
	priv.D = k
	priv.PublicKey.X, priv.PublicKey.Y = p256.ScalarBaseMult(k.Bytes())
	return priv, nil
}

func parseToEcdsaP256PublicKey(pubkey []byte) (*ecdsa.PublicKey, error) {
	if len(pubkey) != 64 {
		return nil, errors.New("parse to ecdsap256 pubkey invalid pubkey")
	}
	pubk := new(ecdsa.PublicKey)
	pubk.Curve = elliptic.P256()
	pubk.X = big.NewInt(0).SetBytes(pubkey[:32])
	pubk.Y = big.NewInt(0).SetBytes(pubkey[32:64])

	return pubk, nil
}

func EcdsaP256Sign(msg, seckey []byte) ([]byte, error) {
	privk, err := parseToEcdsaP256PrivateKey(seckey)
	if err != nil {
		return nil, err
	}

	// warning: ecdsa sign add random before msg
	signature, err := privk.Sign(rand.Reader, msg, nil)
	if err != nil {
		return nil, err
	}

	r, s, err := sm2.SignDataToSignDigit(signature)
	if err != nil {
		return nil, err
	}

	sign := make([]byte, 64)
	copy(sign[:32], r.Bytes())
	copy(sign[32:], s.Bytes())

	return sign, nil
}

func EcdsaP256Verify(msg []byte, pubkey []byte, sign []byte) (bool, error) {
	public, err := parseToEcdsaP256PublicKey(pubkey)
	if err != nil {
		return false, err
	}
	r := big.NewInt(0).SetBytes(sign[:32])
	s := big.NewInt(0).SetBytes(sign[32:64])
	ret := ecdsa.Verify(public, msg, r, s)
	return ret, nil
}

func parseToSM2PrivateKey(seckey []byte) (*sm2.PrivateKey, error) {
	if len(seckey) != 32 {
		return nil, errors.New("invalid seckey")
	}

	c := sm2.P256Sm2()
	k := big.NewInt(0).SetBytes(seckey)
	priv := new(sm2.PrivateKey)
	priv.PublicKey.Curve = c
	priv.D = k
	priv.PublicKey.X, priv.PublicKey.Y = c.ScalarBaseMult(k.Bytes())
	return priv, nil
}

func parseToSM2PublicKey(pubkey []byte) (*sm2.PublicKey, error) {
	if len(pubkey) != 64 {
		return nil, errors.New("parse to sm2 pubkey and invalid pubkey")
	}

	c := sm2.P256Sm2()
	pubk := new(sm2.PublicKey)
	pubk.Curve = c
	pubk.X = big.NewInt(0).SetBytes(pubkey[:32])
	pubk.Y = big.NewInt(0).SetBytes(pubkey[32:64])
	return pubk, nil
}

func SM2Sign(msg, seckey []byte) ([]byte, error) {
	priv, err := parseToSM2PrivateKey(seckey)
	if err != nil {
		return nil, err
	}
	signature, err := priv.Sign(nil, msg, nil)
	if err != nil {
		return nil, err
	}
	r, s, err := sm2.SignDataToSignDigit(signature)
	if err != nil {
		return nil, err
	}
	sign := make([]byte, 64)
	copy(sign[:32], r.Bytes())
	copy(sign[32:], s.Bytes())

	return sign, nil
}

func SM2Verify(msg []byte, pubkey []byte, sign []byte) (bool, error) {
	public, err := parseToSM2PublicKey(pubkey)
	if err != nil {
		return false, err
	}
	r := big.NewInt(0).SetBytes(sign[:32])
	s := big.NewInt(0).SetBytes(sign[32:64])
	signature, err := sm2.SignDigitToSignData(r, s)
	if err != nil {
		return false, err
	}
	ret := public.Verify(msg, signature)
	return ret, nil
}
