package tool

import (
	"ChainGrpcTest/log"
	"context"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/common/math"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	"github.com/xuri/excelize/v2"
	"math/big"
	"sync/atomic"
	"time"
)

func InitAccount(cfg *Config, createAccCount int) {
	if createAccCount <= 0 {
		return
	}
	excelFile := excelize.NewFile()
	index, _ := excelFile.NewSheet("Sheet1")
	// 设置工作簿的默认工作表
	excelFile.SetActiveSheet(index)
	for i := 1; i <= createAccCount; i++ {
		toPrivateKey, err := crypto.GenerateKey()
		if err != nil {
			log.Error("Gen wallet Err:%r", err)
			break
		}
		toPrivateKeyEncode := hexutil.Encode(crypto.FromECDSA(toPrivateKey))[2:]
		toAddress := crypto.PubkeyToAddress(toPrivateKey.PublicKey)
		// 设置单元格的值
		titleSlice := []interface{}{toAddress, toPrivateKeyEncode}
		axis := fmt.Sprintf("A%d", i)
		excelFile.SetSheetRow("Sheet1", axis, &titleSlice)
		log.Infof("Gen count: %d", i)
	}
	//根据指定路径保存文件
	if err := excelFile.SaveAs(cfg.StorageAccFileName); err != nil {
		log.Error("Save address file error: %s", err)
	} else {
		log.Info("Save address file successful")
	}
}

func BathSendTran(cfg *Config, signedTxArr []*types.Transaction, client *ethclient.Client) {
	tranTxCh := make(chan *types.Transaction, 1000000)
	var bathHandleSendCount int64
	startTime := time.Now()
	log.Info("start time:", startTime.String())
	for count := 0; count < cfg.GoRoutineCount; count++ {
		go func() {
			for {
				select {
				case signedTx := <-tranTxCh:
					timeout, cancelFunc := context.WithTimeout(context.Background(), time.Second*3)
					defer cancelFunc()
					err := client.SendTransaction(timeout, signedTx)
					log.Info("Send tx:", signedTx.Hash())
					atomic.AddInt64(&bathHandleSendCount, 1)
					if err != nil {
						log.Error("sendTranErr:", err)
						continue
					}
				}
			}
		}()
	}
	for _, sign := range signedTxArr {
		tranTxCh <- sign
	}
	for {
		if bathHandleSendCount == int64(len(signedTxArr)) {
			log.Infof("Send tran count: %d", bathHandleSendCount)
			log.Info("end time:", time.Now().String())
			log.Info("since time:", time.Since(startTime).Milliseconds())
			break
		}
	}
}

func signTxArr(cfg *Config, amount int64) ([]*types.Transaction, *ethclient.Client) {
	accounts := ReadExcel(cfg.Count, cfg.StorageAccFileName)
	log.Info("Read file successful")
	client, err := ethclient.Dial(cfg.RpcNode)
	if err != nil {
		log.Error("Connect chain error", err)
	}
	log.Info("Connect client successful")
	defer func() {
		client.Close()
	}()
	signerKey, err := crypto.HexToECDSA(cfg.InitAccountPrv)
	fromAddress := crypto.PubkeyToAddress(signerKey.PublicKey)
	nonce, err := client.NonceAt(context.Background(), fromAddress, nil)
	chNonce := big.NewInt(int64(nonce))
	signedTxArr := make([]*types.Transaction, 0)
	for _, rows := range accounts {
		decode, err := hexutil.Decode(rows[0])
		if err != nil {
			continue
		}
		address := common.BytesToAddress(decode)
		txData := types.LegacyTx{
			Nonce:    chNonce.Uint64(),
			To:       &address,
			Value:    math.BigPow(amount*10, 18),
			Gas:      100000,
			GasPrice: big.NewInt(1000000001),
			Data:     nil,
		}
		newtx := types.NewTx(&txData)
		signedTx, err := types.SignTx(newtx, types.NewEIP155Signer(big.NewInt(100)), signerKey)
		if err != nil {
			log.Error("Signed Error", err)
		}
		signedTxArr = append(signedTxArr, signedTx)
		chNonce = chNonce.Add(chNonce, big.NewInt(1))
	}
	return signedTxArr, client
}

// TransferAccount : Perform local currency transfer transactions directly through account signatures
func TransferAccount(cfg *Config, amount int64) {
	signedTxArr, client := signTxArr(cfg, amount)
	BathSendTran(cfg, signedTxArr, client)
}

func ReadExcelOfStartEnd(start int, end int, fileName string) [][]string {
	file, err := excelize.OpenFile(fileName)
	if err != nil {
		log.Error("Open excel err", err.Error())
		return nil
	}
	rows, err := file.GetRows("Sheet1")
	if err != nil {
		log.Error("Read excel failed: " + err.Error())
		return nil
	}
	res := make([][]string, 0)
	for i := start; i <= end; i++ {
		res = append(res, rows[i])
	}
	return res
}

func ReadExcel(count int, fileName string) [][]string {
	file, err := excelize.OpenFile(fileName)
	if err != nil {
		log.Error("Open excel err", err.Error())
		return nil
	}
	rows, err := file.GetRows("Sheet1")
	if err != nil {
		log.Error("Read excel failed: " + err.Error())
		return nil
	}
	res := make([][]string, 0)
	for _, row := range rows {
		if len(res) == count {
			break
		}
		res = append(res, row)
	}
	return res
}
