package main

import (
	"crypto/ecdsa"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"math/big"
)

type Account struct {
	Address common.Address 	`json:"address"`
	Private string 	`json:"private"`
	Nonce   uint64   `json:"nonce"`
	PK      *ecdsa.PrivateKey	`json:"-"`
}

func (acc *Account) MakeInitTx(chainid *big.Int) *types.Transaction {
	unit := new(big.Int).Exp(big.NewInt(10), big.NewInt(18), nil)
	txData := &types.LegacyTx{
		Nonce:    0,
		To:       &acc.Address,
		Value:    new(big.Int).Mul(unit, big.NewInt(10)),
		Gas:      300000,
		GasPrice: big.NewInt(1000000000),
		Data:     nil,
	}
	tx, err := acc.SignTx(types.NewTx(txData), chainid)
	if err != nil {
		return nil
	}
	return tx
}

func (acc *Account) SignTx(tx *types.Transaction, chainid *big.Int) (*types.Transaction, error){
	signedTx, err := types.SignTx(tx, types.NewEIP155Signer(chainid), acc.PK)
	if err != nil {
		return nil, err
	}
	return signedTx, nil
}

func CreateAccounts(count int) []*Account {
	accs := make([]*Account, 0, count)
	for i:=0; i < count; i++ {
		pk,_ := crypto.GenerateKey()
		addr := crypto.PubkeyToAddress(pk.PublicKey)
		private := hexutil.Encode(crypto.FromECDSA(pk))
		accs = append(accs, &Account{Address: addr, Private: private, PK: pk})
	}
	return accs
}
