package transaction

import (
	"ChainGrpcTest/log"
	"ChainGrpcTest/tool"
	"context"
	crypterv1 "github.com/CaduceusMetaverseProtocol/MetaProtocol/gen/proto/go/crypter/v1"
	"github.com/ethereum/go-ethereum/core/types"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"sync/atomic"
	"time"
)

var (
	batchSignRequest                                     chan *crypterv1.BatchSignRequest
	clientMap                                            map[int]crypterv1.CrypterServiceClient
	bathHandleSendCount, allSignedTxCount, totalSendTime int64
)

func init() {
	batchSignRequest = make(chan *crypterv1.BatchSignRequest, 1000000)
	clientMap = make(map[int]crypterv1.CrypterServiceClient)
}

// batchSign 发送的签名交易
func batchSign(client crypterv1.CrypterServiceClient, sleepTime int) {
	var beforeSendTxTime time.Time
	for {
		select {
		case batchSign := <-batchSignRequest:
			if sleepTime != 0 {
				time.Sleep(time.Millisecond * time.Duration(sleepTime))
			}
			sendTranStartTime := time.Now()
			if beforeSendTxTime.UnixMilli() > 0 && time.Since(beforeSendTxTime).Milliseconds() < 1 {
				time.Sleep(time.Millisecond * time.Duration(1))
			}
			sign, err := client.BatchSign(context.Background(), batchSign, grpc.CallContentSubtype(""))
			if err != nil {
				log.Errorf("SendTranErr: %s", err)
				sign, err = client.BatchSign(context.Background(), batchSign, grpc.CallContentSubtype(""))
				if err != nil {
					log.Errorf("Send tran twice err: %s", err)
					continue
				}
			}
			beforeSendTxTime = time.Now()
			atomic.AddInt64(&bathHandleSendCount, 1)
			sinceTime := time.Since(sendTranStartTime).Milliseconds()
			atomic.AddInt64(&totalSendTime, sinceTime)
			log.Infof("Send transaction time: %d ms, sign: %s ", sinceTime, sign.String())
		}
	}
}

// BatchSignHandler 处理批量发送的签名交易
func BatchSignHandler(tranArr []*types.Transaction, cfg *tool.Config) error {
	client, err := grpc.Dial(cfg.RpcNode, grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		log.Error("dial nebula failed", "err", err)
	}
	for i := 0; i < cfg.GoRoutineCount; i++ {
		go batchSign(crypterv1.NewCrypterServiceClient(client), cfg.SleepTime)
	}
	allSignedTxCount = int64(len(tranArr))
	startTime := time.Now()
	for i := 0; i < int(allSignedTxCount); i++ {
		req := crypterv1.BatchSignRequest{}
		batchSignRequest <- &req
	}
	for {
		if bathHandleSendCount == allSignedTxCount {
			log.Infof("Send tran count: %d", bathHandleSendCount)
			log.Infof("Since time: %d ms", time.Since(startTime).Milliseconds())
			break
		}
	}
	return nil
}

// BatchRecover 发送的签名交易
func batchRecover(client crypterv1.CrypterServiceClient, sleepTime int) {
	var beforeSendTxTime time.Time
	for {
		select {
		case batchRecover := <-batchSignRequest:
			if sleepTime != 0 {
				time.Sleep(time.Millisecond * time.Duration(sleepTime))
			}
			sendTranStartTime := time.Now()
			if beforeSendTxTime.UnixMilli() > 0 && time.Since(beforeSendTxTime).Milliseconds() < 1 {
				time.Sleep(time.Millisecond * time.Duration(1))
			}
			sign, err := client.BatchSign(context.Background(), batchRecover, grpc.CallContentSubtype(""))
			if err != nil {
				log.Errorf("SendTranErr: %s", err)
				sign, err = client.BatchSign(context.Background(), batchRecover, grpc.CallContentSubtype(""))
				if err != nil {
					log.Errorf("Send tran twice err: %s", err)
					continue
				}
			}
			beforeSendTxTime = time.Now()
			atomic.AddInt64(&bathHandleSendCount, 1)
			sinceTime := time.Since(sendTranStartTime).Milliseconds()
			atomic.AddInt64(&totalSendTime, sinceTime)
			log.Infof("Send transaction time: %d ms, sign: %s ", sinceTime, sign.String())
		}
	}
}

// BatchRecoverHandler 处理批量发送的签名交易
func BatchRecoverHandler(tranArr []*types.Transaction, cfg *tool.Config) error {
	grpcClient, err := grpc.Dial(cfg.RpcNode, grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		log.Error("dial nebula failed", "err", err)
	}
	serviceClient := crypterv1.NewCrypterServiceClient(grpcClient)
	go batchRecover(serviceClient, cfg.SleepTime)
	allSignedTxCount = int64(len(tranArr))
	startTime := time.Now()
	for i := 0; i < int(allSignedTxCount); i++ {
		req := crypterv1.BatchSignRequest{}
		batchSignRequest <- &req

	}
	for {
		if bathHandleSendCount == allSignedTxCount {
			log.Infof("Send tran count: %d", bathHandleSendCount)
			log.Infof("Since time: %d ms", time.Since(startTime).Milliseconds())
			break
		}
	}
	return nil
}

// BatchVerify 发送的签名交易
func batchVerify(client crypterv1.CrypterServiceClient, sleepTime int) {
	var beforeSendTxTime time.Time
	for {
		select {
		case batchVerify := <-batchSignRequest:
			if sleepTime != 0 {
				time.Sleep(time.Millisecond * time.Duration(sleepTime))
			}
			sendTranStartTime := time.Now()
			if beforeSendTxTime.UnixMilli() > 0 && time.Since(beforeSendTxTime).Milliseconds() < 1 {
				time.Sleep(time.Millisecond * time.Duration(1))
			}
			sign, err := client.BatchSign(context.Background(), batchVerify, grpc.CallContentSubtype(""))
			if err != nil {
				log.Errorf("SendTranErr: %s", err)
				sign, err = client.BatchSign(context.Background(), batchVerify, grpc.CallContentSubtype(""))
				if err != nil {
					log.Errorf("Send tran twice err: %s", err)
					continue
				}
			}
			beforeSendTxTime = time.Now()
			atomic.AddInt64(&bathHandleSendCount, 1)
			sinceTime := time.Since(sendTranStartTime).Milliseconds()
			atomic.AddInt64(&totalSendTime, sinceTime)
			log.Infof("Send transaction time: %d ms, sign: %s ", sinceTime, sign.String())
		}
	}
}

// BatchVerifyHandler 处理批量发送的签名交易
func BatchVerifyHandler(tranArr []*types.Transaction, cfg *tool.Config) error {
	client, err := grpc.Dial(cfg.RpcNode, grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		log.Error("dial nebula failed", "err", err)
	}
	go batchVerify(crypterv1.NewCrypterServiceClient(client), cfg.SleepTime)
	allSignedTxCount = int64(len(tranArr))
	startTime := time.Now()
	for i := 0; i < int(allSignedTxCount); i++ {
		req := crypterv1.BatchSignRequest{}
		batchSignRequest <- &req

	}
	for {
		if bathHandleSendCount == allSignedTxCount {
			log.Infof("Send tran count: %d", bathHandleSendCount)
			log.Infof("Since time: %d ms", time.Since(startTime).Milliseconds())
			break
		}
	}
	return nil
}

// batchRecoverTx 发送的签名交易
func batchRecoverTx(client crypterv1.CrypterServiceClient, sleepTime int) {
	var beforeSendTxTime time.Time
	for {
		select {
		case batchRecoverTx := <-batchSignRequest:
			if sleepTime != 0 {
				time.Sleep(time.Millisecond * time.Duration(sleepTime))
			}
			sendTranStartTime := time.Now()
			if beforeSendTxTime.UnixMilli() > 0 && time.Since(beforeSendTxTime).Milliseconds() < 1 {
				time.Sleep(time.Millisecond * time.Duration(1))
			}
			sign, err := client.BatchSign(context.Background(), batchRecoverTx, grpc.CallContentSubtype(""))
			if err != nil {
				log.Errorf("SendTranErr: %s", err)
				sign, err = client.BatchSign(context.Background(), batchRecoverTx, grpc.CallContentSubtype(""))
				if err != nil {
					log.Errorf("Send tran twice err: %s", err)
					continue
				}
			}
			beforeSendTxTime = time.Now()
			atomic.AddInt64(&bathHandleSendCount, 1)
			sinceTime := time.Since(sendTranStartTime).Milliseconds()
			atomic.AddInt64(&totalSendTime, sinceTime)
			log.Infof("Send transaction time: %d ms, sign: %s ", sinceTime, sign.String())
		}
	}
}

// BatchRecoverTxHandler 处理批量发送的签名交易
func BatchRecoverTxHandler(tranArr []*types.Transaction, cfg *tool.Config) error {
	client, err := grpc.Dial(cfg.RpcNode, grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		log.Error("dial nebula failed", "err", err)
	}
	go batchRecoverTx(crypterv1.NewCrypterServiceClient(client), cfg.SleepTime)
	allSignedTxCount = int64(len(tranArr))
	startTime := time.Now()
	for i := 0; i < int(allSignedTxCount); i++ {
		req := crypterv1.BatchSignRequest{}
		batchSignRequest <- &req

	}
	for {
		if bathHandleSendCount == allSignedTxCount {
			log.Infof("Send tran count: %d", bathHandleSendCount)
			log.Infof("Since time: %d ms", time.Since(startTime).Milliseconds())
			break
		}
	}
	return nil
}
