package transaction

import (
	"ChainGrpcTest/log"
	"ChainGrpcTest/tool"
	"crypto/ecdsa"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"math/big"
	"time"
)

type TranConfig struct {
	Amount                   int64
	GoRoutineCount           int
	ChainId                  int64
	PrivateKey, ReceivedAddr string
	Nonce                    *big.Int
}

type Transactor struct {
	config       TranConfig
	signerKey    *ecdsa.PrivateKey
	sender       common.Address
	receivedAddr common.Address
}

func newTransactor(cfg TranConfig) (*Transactor, error) {
	signerKey, err := crypto.HexToECDSA(cfg.PrivateKey)
	if err != nil {
		log.Error("Error crypto HexToECDSA")
		return nil, err
	}
	// through privateKey get account address
	publicKey := signerKey.Public()
	publicKeyECDSA, ok := publicKey.(*ecdsa.PublicKey)
	if !ok {
		log.Error("Error casting public key to ECDSA")
	}
	address := crypto.PubkeyToAddress(*publicKeyECDSA)
	res := Transactor{
		signerKey:    signerKey,
		receivedAddr: common.HexToAddress(cfg.ReceivedAddr),
		config:       cfg,
		sender:       address,
	}
	return &res, nil
}

// SignedTxArr 获取全部签名数据
func SignedTxArr(sendTxAccountArr [][]string, cfg *tool.Config) []*types.Transaction {
	tranArr := make([]*types.Transaction, 0)
	var signedTx *types.Transaction
	for _, rows := range sendTxAccountArr {
		privateKey := rows[1]
		nonce := big.NewInt(1)
		for signCount := 0; signCount < cfg.SignCount; signCount++ {
			tranCfg := TranConfig{
				Amount:         cfg.Amount,
				ChainId:        cfg.ChainId,
				PrivateKey:     privateKey,
				ReceivedAddr:   cfg.ReceiveAddr,
				GoRoutineCount: cfg.GoRoutineCount,
				Nonce:          nonce,
			}
			t, err := newTransactor(tranCfg)
			signedTx, err = t.signedTx()
			nonce = big.NewInt(1).Add(nonce, big.NewInt(1))
			if err != nil || signedTx == nil {
				log.Errorf("signed tx error %s ", err)
				continue
			}
			tranArr = append(tranArr, signedTx)
		}
	}
	return tranArr
}

// signedTx 签名本币转账交易
func (t *Transactor) signedTx() (*types.Transaction, error) {
	txData := types.LegacyTx{
		Nonce:    t.config.Nonce.Uint64(),
		To:       &t.receivedAddr,
		Value:    big.NewInt(t.config.Amount),
		Gas:      300000,
		GasPrice: big.NewInt(1000000001),
		Data:     nil,
	}
	newtx := types.NewTx(&txData)
	signedTx, err := types.SignTx(newtx, types.NewEIP155Signer(big.NewInt(t.config.ChainId)), t.signerKey)
	if err != nil {
		log.Errorf("Send tx nonce: %d , From: %s , to: %s , error: %s", t.config.Nonce, crypto.PubkeyToAddress(t.signerKey.PublicKey), t.receivedAddr, err.Error())
		time.Sleep(time.Second)
		return nil, err
	}
	return signedTx, nil
}
