package benchmark_test

import (
	"bytes"
	"context"
	"crypto/ecdsa"
	"encoding/gob"
	"errors"
	"fmt"
	"time"
	"unsafe"

	"math/big"
	"net"
	"testing"

	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/anypb"
	"google.golang.org/protobuf/types/known/emptypb"

	base "github.com/CaduceusMetaverseProtocol/MetaProtocol/gen/proto/go/base/v1"
	ring "github.com/CaduceusMetaverseProtocol/MetaProtocol/gen/proto/go/ring/v1"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
)

var countParallelAsEth int64
var countParallelAsStd int64
var countParallelAsAny int64
var countParallelAsBytes int64

type RingServer struct {
	ring.UnimplementedRingServiceServer
}

func (*RingServer) SendRepeatedStdTx(context.Context, *base.RepeatedStdTx) (*ring.SendRawTransactionResponse, error) {

	//fmt.Printf("SendEmptyMsg: %s \n", time.Now())
	return &ring.SendRawTransactionResponse{}, nil
}

func (*RingServer) SendRepeatedEthTx(context.Context, *base.RepeatedEthTx) (*ring.SendRawTransactionResponse, error) {

	//fmt.Printf("SendEmptyMsg: %s \n", time.Now())
	return &ring.SendRawTransactionResponse{}, nil
}

func (*RingServer) SendEmptyMsg(context.Context, *ring.EmptyRequest) (*ring.EmptyResponse, error) {

	//fmt.Printf("SendEmptyMsg: %s \n", time.Now())
	return &ring.EmptyResponse{}, nil
}

func (*RingServer) SendEmpty(context.Context, *emptypb.Empty) (*emptypb.Empty, error) {
	return &emptypb.Empty{}, nil
}

func (*RingServer) SendTxAsEth(ctx context.Context, in *base.TransactionEth) (*ring.SendRawTransactionResponse, error) {

	addr := common.Address{}

	copy(addr[:], in.Tx.Inner.Recipient.Address[:common.AddressLength])

	ethTx := types.NewTx(&types.LegacyTx{
		Nonce:    in.Tx.Inner.AccountNonce,
		To:       &addr,
		Value:    new(big.Int).SetBytes(in.Tx.Inner.Amount),
		Gas:      in.Tx.Inner.GasLimit,
		GasPrice: new(big.Int).SetBytes(in.Tx.Inner.Price),
		Data:     in.Tx.Inner.Payload,
		V:        new(big.Int).SetBytes(in.Tx.Inner.V),
		R:        new(big.Int).SetBytes(in.Tx.Inner.R),
		S:        new(big.Int).SetBytes(in.Tx.Inner.S),
	})

	// atomic.AddInt64(&countParallelAsEth, 1)
	// defer fmt.Println("countParallelAsEth", countParallelAsEth)

	return &ring.SendRawTransactionResponse{TxHash: ethTx.Hash().Bytes()}, nil
}

func (*RingServer) SendTxAsStd(ctx context.Context, in *base.TransactionStd) (*ring.SendRawTransactionResponse, error) {

	addr := common.Address{}

	copy(addr[:], in.Tx.Inner.Recipient.Address[:common.AddressLength])

	ethTx := types.NewTx(&types.LegacyTx{
		Nonce:    in.Tx.Inner.AccountNonce,
		To:       &addr,
		Value:    new(big.Int).SetBytes(in.Tx.Inner.Amount),
		Gas:      in.Tx.Inner.GasLimit,
		GasPrice: new(big.Int).SetBytes(in.Tx.Inner.Price),
		Data:     in.Tx.Inner.Payload,
		V:        new(big.Int).SetBytes(in.Tx.Inner.V),
		R:        new(big.Int).SetBytes(in.Tx.Inner.R),
		S:        new(big.Int).SetBytes(in.Tx.Inner.S),
	})

	// atomic.AddInt64(&countParallelAsStd, 1)

	// defer fmt.Println("countParallelAsStd", countParallelAsStd)

	return &ring.SendRawTransactionResponse{TxHash: ethTx.Hash().Bytes()}, nil
}

func (*RingServer) SendTxAsAny(ctx context.Context, in *base.Transaction) (*ring.SendRawTransactionResponse, error) {

	msg, err := in.Tx.UnmarshalNew()

	if err != nil {
		return nil, err
	}

	// atomic.AddInt64(&countParallelAsAny, 1)
	// defer fmt.Println("countParallelAsAny", countParallelAsAny)

	switch m := msg.(type) {

	case *base.EthTx:

		addr := common.Address{}

		copy(addr[:], m.Inner.Recipient.Address[:common.AddressLength])

		ethTx := types.NewTx(&types.LegacyTx{
			Nonce:    m.Inner.AccountNonce,
			To:       &addr,
			Value:    new(big.Int).SetBytes(m.Inner.Amount),
			Gas:      m.Inner.GasLimit,
			GasPrice: new(big.Int).SetBytes(m.Inner.Price),
			Data:     m.Inner.Payload,
			V:        new(big.Int).SetBytes(m.Inner.V),
			R:        new(big.Int).SetBytes(m.Inner.R),
			S:        new(big.Int).SetBytes(m.Inner.S),
		})

		return &ring.SendRawTransactionResponse{TxHash: ethTx.Hash().Bytes()}, nil

	case *base.StdTx:
		addr := common.Address{}

		copy(addr[:], m.Inner.Recipient.Address[:common.AddressLength])

		ethTx := types.NewTx(&types.LegacyTx{
			Nonce:    m.Inner.AccountNonce,
			To:       &addr,
			Value:    new(big.Int).SetBytes(m.Inner.Amount),
			Gas:      m.Inner.GasLimit,
			GasPrice: new(big.Int).SetBytes(m.Inner.Price),
			Data:     m.Inner.Payload,
			V:        new(big.Int).SetBytes(m.Inner.V),
			R:        new(big.Int).SetBytes(m.Inner.R),
			S:        new(big.Int).SetBytes(m.Inner.S),
		})

		return &ring.SendRawTransactionResponse{TxHash: ethTx.Hash().Bytes()}, nil

	case *base.RepeatedEthTx:

		return &ring.SendRawTransactionResponse{}, nil

	default:
		return nil, errors.New(fmt.Sprintf("server expected tx type: %T and %T, but actually is %T", base.EthTx{}, base.StdTx{}, msg))
	}

}

func (*RingServer) SendTxAsBytes(ctx context.Context, in *base.TransactionBytes) (*ring.SendRawTransactionResponse, error) {

	ethTx := types.Transaction{}

	if err := ethTx.UnmarshalBinary(in.Tx); err != nil {
		return nil, err
	}

	// atomic.AddInt64(&countParallelAsBytes, 1)
	// defer fmt.Printf("countParallelAsBytes: %d\n", countParallelAsBytes)

	return &ring.SendRawTransactionResponse{TxHash: ethTx.Hash().Bytes()}, nil
}

func TestGrpcServer(t *testing.T) {

	lis, err := net.Listen("tcp", ":9006")
	if err != nil {
		t.Fatal(err)
	}

	s := grpc.NewServer()

	ring.RegisterRingServiceServer(s, &RingServer{})

	err = s.Serve(lis)
	if err != nil {
		t.Fatal(err)
	}
}

func TestType(t *testing.T) {

	ethTx := base.EthTx{}

	fmt.Println(ethTx.ProtoReflect().Descriptor().FullName())

}

func TestAnyTx(t *testing.T) {

	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, _ := crypto.GenerateKey()

	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		t.Fatal(err)
	}

	conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
	if err != nil {
		t.Fatal(err)
	}
	defer conn.Close()

	c := ring.NewRingServiceClient(conn)

	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	_, _ = c, ctx

	inner := base.EthTxData{
		AccountNonce: tx.Nonce(),
		Price:        tx.GasPrice().Bytes(),
		GasLimit:     tx.Gas(),
		Payload:      tx.Data(),
	}

	v, r, sigs := tx.RawSignatureValues()

	inner.V = v.Bytes()
	inner.R = r.Bytes()
	inner.S = sigs.Bytes()
	inner.Amount = tx.Value().Bytes()

	addr := base.Address{Address: tx.To().Bytes()}

	inner.Recipient = &addr
	inner.From = fromAddr.Bytes()

	ethTx := base.EthTx{Inner: &inner}

	ethTxAsAny, err := anypb.New(&ethTx)

	if err != nil {
		t.Fatal(err)
	}

	//fmt.Println("ethTxAsAny.ProtoReflect().Descriptor().FullName()", ethTx.ProtoReflect().Descriptor().FullName())
	res, err := c.SendTxAsAny(ctx, &base.Transaction{Tx: ethTxAsAny})

	_ = res
	if err != nil {
		t.Fatal(err)
	}

	stdInner := base.StdTxData{
		AccountNonce: tx.Nonce(),
		Price:        tx.GasPrice().Bytes(),
		GasLimit:     tx.Gas(),
		Payload:      tx.Data(),
	}

	// v, r, sigs := tx.RawSignatureValues()

	stdInner.V = v.Bytes()
	stdInner.R = r.Bytes()
	stdInner.S = sigs.Bytes()
	stdInner.Amount = tx.Value().Bytes()

	//addr := base.Address{Address: tx.To().Bytes()}

	stdInner.Recipient = &addr

	stdTx := base.StdTx{Inner: &stdInner}

	_ = stdTx

	stdTxAsAny, err := anypb.New(&stdTx)

	if err != nil {
		t.Fatal(err)
	}

	_ = stdTxAsAny

	res, err = c.SendTxAsAny(ctx, &base.Transaction{Tx: stdTxAsAny})

	if err != nil {
		t.Fatal(err)
	}

	_ = res

}

func TestSizeOf(t *testing.T) {

	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, _ := crypto.GenerateKey()

	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		t.Fatal(err)
	}

	inner := base.EthTxData{
		AccountNonce: tx.Nonce(),
		Price:        tx.GasPrice().Bytes(),
		GasLimit:     tx.Gas(),
		Payload:      tx.Data(),
	}

	v, r, sigs := tx.RawSignatureValues()

	inner.V = v.Bytes()
	inner.R = r.Bytes()
	inner.S = sigs.Bytes()
	inner.Amount = tx.Value().Bytes()

	addr := base.Address{Address: tx.To().Bytes()}

	inner.Recipient = &addr
	inner.From = fromAddr.Bytes()

	out, err := proto.Marshal(&base.TransactionEth{Tx: &base.EthTx{Inner: &inner}})

	if err != nil {
		t.Fatal(err)
	}

	fmt.Printf("TransactionEth proto buff encoding: %d \n", len(out))

	//fmt.Printf("%T \n", unsafe.Sizeof(tx.Nonce()))

	sss, err := getRealSizeOf(tx)

	if err != nil {
		t.Fatal(err)
	}
	fmt.Println("getRealSizeOf tx ", sss)

	ssss, err := getRealSizeOf(base.TransactionEth{Tx: &base.EthTx{Inner: &inner}})

	if err != nil {
		t.Fatal(err)
	}
	fmt.Println("getRealSizeOf TransactionEth", ssss)

	fmt.Println("unsafe.Sizeof", unsafe.Sizeof(*tx))
	fmt.Println("unsafe.Sizeof v", unsafe.Sizeof(*v))
	fmt.Println("unsafe.Sizeof r", unsafe.Sizeof(*r))
	fmt.Println("unsafe.Sizeof sigs", unsafe.Sizeof(*sigs))

	fmt.Println("tx size", tx.Size())

	// v, r, sigs

}

// type ASDF struct {
//     A uint64
//     B uint64
//     C uint64
//     D uint64
//     E uint64
//     F string
// }

// func (s *ASDF) size() int {
//     size := int(unsafe.Sizeof(*s))
//     size += len(s.F)
//     return size
// }

func getRealSizeOf(v interface{}) (int, error) {
	b := new(bytes.Buffer)
	if err := gob.NewEncoder(b).Encode(v); err != nil {
		return 0, err
	}
	return b.Len(), nil
}
