package types

import (
	"errors"
)

var (
	ErrChanIsFull     = errors.New("channel is full")
	ErrChanClosed     = errors.New("channel is closed")
	ErrRequestTimeout = errors.New("request timeout")
)

type FeatureId byte

const (
	FeatureBase FeatureId = iota
	FeatureSecp256k1RecoveryPubkey
	FeatureSecp256k1RecoverableSign
	FeatureSecp256k1Sign
	FeatureSecp256k1Verify
	FeatureEcdsaP256Sign
	FeatureEcdsaP256Verify
	FeatureSM2Sign
	FeatureSM2Verify
)

type XTask interface {
	TaskName() string
	TaskId() FeatureId
}

type XTaskSecp256k1RPubkey struct {
	Msg    []byte
	Rsig   []byte
	Pubkey []byte
	Address []byte
}

func (x *XTaskSecp256k1RPubkey) TaskName() string {
	return "Secp256K1-Recover-Pubkey"
}
func (x *XTaskSecp256k1RPubkey) TaskId() FeatureId {
	return FeatureSecp256k1RecoveryPubkey
}

type XTaskSecp256k1RSign struct {
	Msg   []byte
	Privk []byte
	Rsig  []byte
}

func (x *XTaskSecp256k1RSign) TaskName() string {
	return "Secp256K1-Recoverable-Sign"
}

func (x *XTaskSecp256k1RSign) TaskId() FeatureId {
	return FeatureSecp256k1RecoverableSign
}

type XTaskSecp256k1Sign struct {
	Msg   []byte
	Privk []byte
	Sig   []byte
}

func (x *XTaskSecp256k1Sign) TaskName() string {
	return "Secp256K1-Sign"
}

func (x *XTaskSecp256k1Sign) TaskId() FeatureId {
	return FeatureSecp256k1Sign
}

type XTaskSecp256k1Verify struct {
	Msg    []byte
	Sig    []byte
	Pubkey []byte
	Verify bool
}

func (x *XTaskSecp256k1Verify) TaskName() string {
	return "Secp256K1-Verify"
}

func (x *XTaskSecp256k1Verify) TaskId() FeatureId {
	return FeatureSecp256k1Verify
}

type XTaskEcdsa256Sign struct {
	Msg   []byte
	Privk []byte
	Sig   []byte
}

func (x *XTaskEcdsa256Sign) TaskName() string {
	return "EcdsaP256-Sign"
}

func (x *XTaskEcdsa256Sign) TaskId() FeatureId {
	return FeatureEcdsaP256Sign
}

type XTaskEcdsa256Verify struct {
	Msg    []byte
	Sig    []byte
	Pubkey []byte
	Verify bool
}

func (x *XTaskEcdsa256Verify) TaskName() string {
	return "EcdsaP256-Verify"
}

func (x *XTaskEcdsa256Verify) TaskId() FeatureId {
	return FeatureEcdsaP256Verify
}

type XTaskSM2P256Sign struct {
	Msg   []byte
	Privk []byte
	Sig   []byte
}

func (x *XTaskSM2P256Sign) TaskName() string {
	return "SM2P256-Sign"
}

func (x *XTaskSM2P256Sign) TaskId() FeatureId {
	return FeatureSM2Sign
}

type XTaskSM2P256Verify struct {
	Msg    []byte
	Sig    []byte
	Pubkey []byte
	Verify bool
}

func (x *XTaskSM2P256Verify) TaskName() string {
	return "SM2P256-Verify"
}

func (x *XTaskSM2P256Verify) TaskId() FeatureId {
	return FeatureSM2Verify
}

type TaskWithReport struct {
	XTask
	ch chan XTask
}

func (t *TaskWithReport) Report() error {
	return SafeWriteXTask(t.ch, t.XTask)
}

func MakeTaskWithReport(task XTask, report chan XTask) *TaskWithReport {
	return &TaskWithReport{
		task,
		report,
	}
}

type Engine interface {
	Name() string
	Ready() bool
	Support(XTask) bool

	Process(XTask) (XTask, error)
	ProcessA(*TaskWithReport) error
	ProcessBatch([]XTask) ([]XTask, error)
}
