package cpu

import (
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/crypto"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/xecc/engine"
	. "github.com/CaduceusMetaverseProtocol/MetaCryptor/xecc/types"
	"github.com/CaduceusMetaverseProtocol/MetaCryptor/common/log"
	"sync"
)

var (
	cpuEngine = &CPUEngine{}
)

type CPUEngine struct {
	softTask chan *TaskWithReport
	closed   chan struct{}
	ready    bool
	mux      sync.Mutex
	initOnce sync.Once
}

func GetInstance() *CPUEngine {
	cpuEngine.mux.Lock()
	defer cpuEngine.mux.Unlock()

	cpuEngine.initOnce.Do(func() {
		cpuEngine.softTask = make(chan *TaskWithReport, 100000)
		cpuEngine.closed = make(chan struct{})
		cpuEngine.ready = true
		go cpuEngine.softtask()
	})
	return cpuEngine
}

func (ce CPUEngine) Ready() bool {
	return ce.ready
}

func (ce CPUEngine) Support(task XTask) bool {
	return true
}

func (ce CPUEngine) Name() string {
	return "cpu-engine"
}

func (ce CPUEngine) Process(task XTask) (XTask, error) {
	if e := ce.doTask(task); e != nil {
		return task, e
	}
	return task, nil
}

func (ce CPUEngine) ProcessA(twp *TaskWithReport) error {
	if ce.Ready() {
		return SafeWriteXTaskWithReport(ce.softTask, twp)
	}
	return engine.ErrNotReady
}

func (ce CPUEngine) ProcessBatch(tasks []XTask) ([]XTask, error) {
	success := make([]XTask, 0, len(tasks))
	failed := make([]XTask, 0, len(tasks))

	sch := make(chan XTask)
	fch := make(chan XTask)

	for _, t := range tasks {
		go func(task XTask) {
			e := ce.doTask(task)
			if e == nil {
				SafeWriteXTask(sch, task)
			} else {
				SafeWriteXTask(fch, task)
			}
		}(t)
	}
	for {
		select {
		case n, ok := <-sch:
			if !ok {
				break
			}
			success = append(success, n)
		case f, ok := <-fch:
			if !ok {
				break
			}
			failed = append(failed, f)
		}

		if (len(success) + len(failed)) == len(tasks) {
			break
		}
	}
	return success, nil
}

func (ce CPUEngine) doTask(task XTask) error {
	var err error
	switch t := task.(type) {
	case *XTaskSecp256k1RPubkey:
		log.Info("recover pubkey with v", t.Rsig[64])
		t.Pubkey, err = crypto.RecoverPubkey(t.Msg, t.Rsig)
		if err != nil {
			log.Error("recover pubkey failed", "err=", err)
		} else {
			t.Address = crypto.Keccak256(t.Pubkey[1:])[12:]
		}

	case *XTaskSecp256k1RSign:
		t.Rsig, err = crypto.RecoverableSign(t.Msg, t.Privk)

	case *XTaskSecp256k1Sign:
		t.Sig, err = crypto.Secp256k1Sign(t.Msg, t.Privk)

	case *XTaskSecp256k1Verify:
		t.Verify, err = crypto.Secp256k1Verify(t.Msg, t.Pubkey, t.Sig)

	case *XTaskEcdsa256Sign:
		t.Sig, err = crypto.EcdsaP256Sign(t.Msg, t.Privk)

	case *XTaskEcdsa256Verify:
		t.Verify, err = crypto.EcdsaP256Verify(t.Msg, t.Pubkey, t.Sig)

	case *XTaskSM2P256Sign:
		t.Sig, err = crypto.SM2Sign(t.Msg, t.Privk)

	case *XTaskSM2P256Verify:
		t.Verify, err = crypto.SM2Verify(t.Msg, t.Pubkey, t.Sig)

	default:
		err = engine.ErrUnsupport
	}
	return err
}

func (ce CPUEngine) softtask() {
	for {
		select {
		case s, ok := <-ce.softTask:
			if !ok {
				return
			}
			t := s.XTask
			if e := ce.doTask(t); e == nil {
				s.Report()
			}
		case <-ce.closed:
			return
		}
	}
}
