package transaction

import (
	"ChainGrpcTest/log"
	"ChainGrpcTest/tool"
	"ChainGrpcTest/txcache"
	"context"
	"crypto/ecdsa"
	ring "github.com/CaduceusMetaverseProtocol/MetaProtocol/gen/proto/go/ring/v1"
	metatypes "github.com/CaduceusMetaverseProtocol/MetaTypes/types"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"google.golang.org/grpc"
	"math/big"
	"net"
	"strconv"
	"strings"
	"sync"
	"sync/atomic"
	"time"
)

type TranConfig struct {
	Amount                   int64
	GoRoutineCount           int
	ChainId                  int64
	PrivateKey, ReceivedAddr string
	Nonce                    *big.Int
}

type Transactor struct {
	config       TranConfig
	signerKey    *ecdsa.PrivateKey
	sender       common.Address
	receivedAddr common.Address
}

type SignTranArr struct {
	TranArr []*types.Transaction
	mux     sync.RWMutex
}

var (
	tran        = make(chan *Transactor, 0)
	signTranArr = &SignTranArr{
		TranArr: make([]*types.Transaction, 0),
	}
	batchSignCount, handleNonceCount int32
)

func newTransactor(cfg TranConfig) (*Transactor, error) {
	signerKey, err := crypto.HexToECDSA(cfg.PrivateKey)
	if err != nil {
		log.Error("Error crypto HexToECDSA")
		return nil, err
	}
	// through privateKey get account address
	publicKey := signerKey.Public()
	publicKeyECDSA, ok := publicKey.(*ecdsa.PublicKey)
	if !ok {
		log.Error("Error casting public key to ECDSA")
	}
	address := crypto.PubkeyToAddress(*publicKeyECDSA)
	res := Transactor{
		signerKey:    signerKey,
		receivedAddr: common.HexToAddress(cfg.ReceivedAddr),
		config:       cfg,
		sender:       address,
	}
	return &res, nil
}

func dialerFunc(ctx context.Context, addr string) (net.Conn, error) {
	return Connect(addr)
}

func Connect(protoAddr string) (net.Conn, error) {
	proto, address := ProtocolAndAddress(protoAddr)
	conn, err := net.Dial(proto, address)
	return conn, err
}

func ProtocolAndAddress(listenAddr string) (string, string) {
	protocol, address := "tcp", listenAddr
	parts := strings.SplitN(address, "://", 2)
	if len(parts) == 2 {
		protocol, address = parts[0], parts[1]
	}
	return protocol, address
}

func InitAccNonce(sendTxAccountArr [][]string, cfg *tool.Config) (error, sync.Map) {
	var accountsNonceMap sync.Map
	rowsCh := make(chan []string, 1000000)
	client, err := grpc.Dial(cfg.RpcNode, grpc.WithInsecure(), grpc.WithContextDialer(dialerFunc))
	if err != nil {
		log.Error("grpc dial error:", err)
		return err, sync.Map{}
	}
	defer client.Close()
	serviceClient := ring.NewRingServiceClient(client)
	for i := 0; i < cfg.GoRoutineCount; i++ {
		go func() {
			for {
				select {
				case sendTxAccount := <-rowsCh:
					addressRow := sendTxAccount[0]
					privateKey := sendTxAccount[1]
					fromAddr := metatypes.HexToAddress(addressRow)
					nonceReq := &ring.NonceRequest{
						Address: (*metatypes.Address)(fromAddr.Bytes()),
					}
					ctx, cancel := context.WithTimeout(context.Background(), time.Second*20)
					defer cancel()
					response, err := serviceClient.Nonce(ctx, nonceReq)
					if err != nil {
						log.Error("get account nonce error:", err)
					}
					intNum, _ := strconv.Atoi(strconv.FormatUint(response.Nonce, 10))
					accountsNonceMap.Store(privateKey, intNum)
					atomic.AddInt32(&handleNonceCount, 1)
				}
			}
		}()
	}
	for _, rows := range sendTxAccountArr {
		rowsCh <- rows
	}
	for {
		if handleNonceCount == int32(len(sendTxAccountArr)) {
			log.Info("Wait get all acc nonce successful")
			return nil, accountsNonceMap
		}
	}
}

// SignedTxArr 获取全部签名数据
func SignedTxArr(syncMap sync.Map, sendTxAccountArr [][]string, cfg *tool.Config) []*types.Transaction {
	for i := 0; i < cfg.GoRoutineCount; i++ {
		go signedTxFunc()
	}
	for _, rows := range sendTxAccountArr {
		fromAddr := rows[0]
		privateKey := rows[1]
		value, ok := syncMap.Load(privateKey)
		if !ok {
			log.Error("Load nonce map error...........")
			continue
		}
		nonce := new(big.Int).SetInt64(int64(value.(int)))
		log.Infof("from addr:%s,nonce:%d", fromAddr, nonce)
		for signCount := 0; signCount < cfg.SignCount; signCount++ {
			tranCfg := TranConfig{
				Amount:         cfg.Amount,
				ChainId:        cfg.ChainId,
				PrivateKey:     privateKey,
				ReceivedAddr:   cfg.ReceiveAddr,
				GoRoutineCount: cfg.GoRoutineCount,
				Nonce:          nonce,
			}
			t, err := newTransactor(tranCfg)
			if err != nil {
				log.Errorf("signed tx error %s ", err)
				continue
			}
			tran <- t
			nonce = big.NewInt(1).Add(nonce, big.NewInt(1))
		}
	}
	for {
		if len(sendTxAccountArr)*cfg.SignCount == int(batchSignCount) && len(signTranArr.TranArr) == len(sendTxAccountArr)*cfg.SignCount {
			batchSignCount = 0
			newTranArr := make([]*types.Transaction, 0)
			for _, tran := range signTranArr.TranArr {
				newTranArr = append(newTranArr, tran)
			}
			signTranArr.TranArr = make([]*types.Transaction, 0)
			return newTranArr
		}
	}
}

// signedTxFunc 签名本币转账交易
func signedTxFunc() (*types.Transaction, error) {
	for {
		select {
		case t := <-tran:
			txData := types.LegacyTx{
				Nonce:    t.config.Nonce.Uint64(),
				To:       &t.receivedAddr,
				Value:    big.NewInt(t.config.Amount),
				Gas:      300000,
				GasPrice: big.NewInt(1000000001),
				Data:     nil,
			}
			newtx := types.NewTx(&txData)
			signedTx, err := types.SignTx(newtx, types.NewEIP155Signer(big.NewInt(t.config.ChainId)), t.signerKey)
			if err != nil {
				log.Errorf("Send tx nonce: %d , From: %s , to: %s , error: %s", t.config.Nonce, crypto.PubkeyToAddress(t.signerKey.PublicKey), t.receivedAddr, err.Error())
				time.Sleep(time.Second)
				return nil, err
			}
			txcache.Add(signedTx.Hash().Hex(), t.sender.Hex())
			atomic.AddInt32(&batchSignCount, 1)
			Add(signedTx)
		}
	}
}

func Add(signedTx *types.Transaction) {
	signTranArr.mux.Lock()
	defer signTranArr.mux.Unlock()
	signTranArr.TranArr = append(signTranArr.TranArr, signedTx)
}

func NoSignTxArr(syncMap sync.Map, sendTxAccountArr [][]string, cfg *tool.Config) []*types.Transaction {
	for _, rows := range sendTxAccountArr {
		fromAddr := rows[0]
		privateKey := rows[1]
		value, ok := syncMap.Load(privateKey)
		if !ok {
			log.Error("Load nonce map error...........")
			continue
		}
		nonce := new(big.Int).SetInt64(int64(value.(int)))
		log.Infof("from addr:%s,nonce:%d", fromAddr, nonce)
		receiveAddr := common.HexToAddress(cfg.ReceiveAddr)
		for signCount := 0; signCount < cfg.SignCount; signCount++ {
			var txData = types.LegacyTx{
				Nonce:    nonce.Uint64(),
				To:       &receiveAddr,
				Value:    big.NewInt(cfg.Amount),
				Gas:      300000,
				GasPrice: big.NewInt(1000000001),
				Data:     nil,
			}
			newtx := types.NewTx(&txData)
			txcache.Add(newtx.Hash().Hex(), fromAddr)
			signTranArr.TranArr = append(signTranArr.TranArr, newtx)
			nonce = big.NewInt(1).Add(nonce, big.NewInt(1))
		}
	}
	return signTranArr.TranArr
}
