package operator

import (
	"context"
	"contract-case/constant"
	"contract-case/contract/coin"
	"contract-case/contract/deploy"
	erc20 "contract-case/contract_abi/erc20_transfer/compile"
	erc721 "contract-case/contract_abi/erc721_transfer/compile"
	"contract-case/log"
	"contract-case/tool"
	"contract-case/util"
	"crypto/ecdsa"
	"encoding/json"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"os"
	"sync"
	"time"
)

//var tranChanel = make(chan *types.Transaction, 1000)

func GenerateAcc(count int) *tool.AccArr {
	fromAddr := make([]string, 0)
	fromPrv := make([]string, 0)
	toAddr := make([]string, 0)
	toPrv := make([]string, 0)
	for i := 0; i < count; i++ {

		toPrivateKey, err := crypto.GenerateKey()
		if err != nil {
			log.Error("Gen wallet Err:%r", err)
			break
		}
		prv := hexutil.Encode(crypto.FromECDSA(toPrivateKey))
		addr := crypto.PubkeyToAddress(toPrivateKey.PublicKey).Hex()
		if i < count/2 {
			fromAddr = append(fromAddr, addr)
			fromPrv = append(fromPrv, prv)
		} else {
			toAddr = append(toAddr, addr)
			toPrv = append(toPrv, prv)
		}
	}
	accArr := &tool.AccArr{
		FromAddr: fromAddr,
		FromPrv:  fromPrv,
		ToAddr:   toAddr,
		ToPrv:    toPrv,
	}
	jsonData, err := json.MarshalIndent(accArr, "", "  ")
	if err != nil {
		log.Error("JSON encoding account failed: ", err)
	}
	err = os.WriteFile("./config/account.json", jsonData, 0644)
	if err != nil {
		fmt.Println("File writing account failed: ", err)
	}
	return accArr
}

func DeployContract(client *ethclient.Client, txRes []*types.Transaction) []*types.Transaction {
	auth, err := bind.NewKeyedTransactorWithChainID(tool.Cfg.DeployPrv, tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	auth.NoSend = true
	auth.Value = big.NewInt(0)
	auth.GasLimit = uint64(4000000)
	auth.GasPrice = big.NewInt(4000000)
	nonce, err := client.NonceAt(context.Background(), tool.Cfg.DeployAddr, nil)
	if err != nil {
		log.Error("DeployTokenTransfer func get acc nonce err:", err)
	}
	contractArr := make([]map[string]common.Address, 0)
	for i := 0; i < 10; i++ {
		contractMap := make(map[string]common.Address, 0)
		for _, deployFunc := range deploy.DeployFunc {
			auth.Nonce = big.NewInt(int64(nonce))
			contractAddr, transaction, contractType := deployFunc(auth, client)
			log.Info("contractAddr:", contractAddr, ", nonce:", nonce)
			txRes = append(txRes, transaction)
			contractMap[contractType] = contractAddr
			nonce += 1
		}
		contractArr = append(contractArr, contractMap)
	}
	util.GenerateContractMap(contractArr)
	return txRes
}

func InitAccountCoin(txArr []*types.Transaction, nonce uint64, amount *big.Int, accArr *tool.AccArrFormat) ([]*types.Transaction, uint64) {
	//初始化From账户
	trade := &coin.Trade{
		FromPrv:   tool.Cfg.DeployPrv,
		FromNonce: big.NewInt(int64(nonce)),
		Amount:    amount,
		Gas:       big.NewInt(50000),
		GasPrice:  big.NewInt(10000000000),
		ChainId:   tool.Cfg.ChainId,
	}
	for i := 0; i < len(accArr.FromAddr); i++ {
		toAddress := accArr.FromAddr[i]
		trade.ToAddr = &toAddress
		tx, err := trade.CoinTransferSignTx()
		if err != nil {
			log.Error("Init account - sign tran err:", err.Error())
			return nil, 0
		}
		txArr = append(txArr, tx)
		nonce += 1
		trade.FromNonce = big.NewInt(int64(nonce))
		log.Info("Init acc successful,addr:", toAddress)
	}
	for i := 0; i < len(accArr.ToAddr); i++ {
		toAddress := accArr.ToAddr[i]
		trade.ToAddr = &toAddress
		tx, err := trade.CoinTransferSignTx()
		if err != nil {
			log.Error("Init account - sign tran err:", err.Error())
			return nil, 0
		}
		txArr = append(txArr, tx)
		nonce += 1
		trade.FromNonce = big.NewInt(int64(nonce))
		log.Info("Init acc successful,addr:", toAddress)
	}
	return txArr, nonce
}

func InitContractCoin(txArr []*types.Transaction, nonce uint64, contractLen int, contractMap []map[string]common.Address, trade *coin.Trade) ([]*types.Transaction, uint64) {
	for i := 0; i < contractLen; i++ {
		trade.FromNonce = big.NewInt(int64(nonce))
		coinTransferContract := contractMap[i][constant.COIN_TRANSFER]
		trade.ToAddr = &coinTransferContract
		tx, err := trade.CoinTransferSignTx()
		if err != nil {
			log.Error("Init account - sign tran err:", err.Error())
			return nil, 0
		}
		txArr = append(txArr, tx)
		nonce += 1
	}
	return txArr, nonce
}

func InitErc721AccMint(txArr []*types.Transaction, nonce uint64, contractLen int, contractMap []map[string]common.Address, accArr *tool.AccArrFormat, client *ethclient.Client, auth *bind.TransactOpts) ([]*types.Transaction, uint64) {
	tx := &types.Transaction{}
	for j := 0; j < contractLen; j++ {
		newERC721, err := erc721.NewERC721(contractMap[j][constant.ERC721], client)
		if err != nil {
			log.Error("Init account - new erc721 err:", err.Error())
			return nil, 0
		}
		tokenIds := make([]*big.Int, 0)
		for i := 0; i < len(accArr.FromAddr); i++ {
			tokenIds = append(tokenIds, big.NewInt(int64(i)))
		}
		auth.GasLimit = 10000000
		auth.GasPrice = big.NewInt(10000000)
		auth.Nonce = big.NewInt(int64(nonce))
		of, err := newERC721.BalanceOf(&bind.CallOpts{}, tool.Cfg.DeployAddr)
		if err != nil {
			return nil, 0
		}
		log.Info("Erc721 Mint before deploy address balance of", tool.Cfg.DeployAddr, " is: ", of.String())
		tx, err = newERC721.BatchMint(auth, tool.Cfg.DeployAddr, tokenIds)
		if err != nil {
			log.Error("Init account - erc721 mint err:", err.Error())
			return nil, 0
		}
		txArr = append(txArr, tx)
		log.Info("Init account - mint 721 successful,tx:", tx.Hash().Hex())
		nonce += 1
	}
	return txArr, nonce
}

func InitErc721AccTransfer(txArr []*types.Transaction, nonce uint64, contractLen int, contractMap []map[string]common.Address, accArr *tool.AccArrFormat, client *ethclient.Client, auth *bind.TransactOpts) ([]*types.Transaction, uint64) {
	tx := &types.Transaction{}
	for j := 0; j < contractLen; j++ {
		newERC721, err := erc721.NewERC721(contractMap[j][constant.ERC721], client)
		if err != nil {
			log.Error("Init account - new erc721 err:", err.Error())
			return nil, 0
		}
		for i := 0; i < len(accArr.FromAddr); i++ {
			auth.Nonce = big.NewInt(int64(nonce))
			tx, err = newERC721.TransferFrom(auth, tool.Cfg.DeployAddr, accArr.FromAddr[i], big.NewInt(int64(i)))
			if err != nil {
				log.Error("Init account - erc721 TransferFrom err:", err.Error())
				return nil, 0
			}
			txArr = append(txArr, tx)
			log.Info("Init account - transferFrom 721 successful,tx:", tx.Hash().Hex())
			nonce += 1

			auth.Nonce = big.NewInt(int64(nonce))
			tx, err = newERC721.SetApprovalForAllFrom(auth, accArr.FromAddr[i], contractMap[j][constant.TOKEN721_TRANSFER], true)
			if err != nil {
				log.Error("Init account - erc721 setApprovalForAllFrom err:", err.Error())
				return nil, 0
			}
			txArr = append(txArr, tx)
			log.Info("Init account - transferFrom 721 successful,tx:", tx.Hash().Hex())
			nonce += 1
		}
	}
	return txArr, nonce
}

func InitErc20Acc(txArr []*types.Transaction, nonce uint64, contractLen int, contractMap []map[string]common.Address, accArr *tool.AccArrFormat, client *ethclient.Client, auth *bind.TransactOpts) ([]*types.Transaction, uint64) {
	tx := &types.Transaction{}
	for j := 0; j < contractLen; j++ {
		newERC20, err := erc20.NewERC20(contractMap[j][constant.ERC20], client)
		if err != nil {
			log.Error("Init account - new erc20 err:", err.Error())
			return nil, 0
		}
		auth.Nonce = big.NewInt(int64(nonce))
		auth.GasPrice = big.NewInt(1000000000)
		auth.GasLimit = big.NewInt(1000000000).Uint64()
		// mint
		of, err := newERC20.BalanceOf(&bind.CallOpts{}, tool.Cfg.DeployAddr)
		if err != nil {
			log.Error(tool.Cfg.DeployAddr, "Get erc20 balance of error:", err.Error())
			return nil, 0
		}
		log.Info(tool.Cfg.DeployAddr, ",erc20 balance of:", of.Uint64())
		if of.Int64() == 0 {
			mintCount := big.NewInt(0)
			mintCount.SetString("1000000000000000000000000", 10)
			tx, err = newERC20.Mint(auth, tool.Cfg.DeployAddr, mintCount)
			if err != nil {
				log.Error("Init account - erc20 mint err:", err.Error())
				return nil, 0
			}
			txArr = append(txArr, tx)
			nonce += 1
			auth.Nonce = big.NewInt(int64(nonce))
		}
		tranCount := big.NewInt(0)
		tranCount.SetString("1000000000000000000000", 10)
		// erc20 转账
		tx, err = newERC20.Transfer(auth, contractMap[j][constant.TOKEN20_TRANSFER], tranCount)
		if err != nil {
			log.Error("Init account - erc20 transfer err:", err.Error())
			return nil, 0
		}
		txArr = append(txArr, tx)
		nonce += 1
		tranCount.SetString("10000000000000000000", 10)
		for i := 0; i < len(accArr.FromAddr); i++ {
			owner := accArr.FromAddr[i]
			auth.Nonce = big.NewInt(int64(nonce))
			tx, err = newERC20.Transfer(auth, owner, tranCount)
			if err != nil {
				log.Error("Init account - erc20 transfer err:", err.Error())
				return nil, 0
			}
			txArr = append(txArr, tx)
			nonce += 1

			auth.Nonce = big.NewInt(int64(nonce))
			tx, err = newERC20.ApproveFrom(auth, owner, contractMap[j][constant.TOKEN20_TRANSFER], tranCount)
			if err != nil {
				log.Error("Init account - erc20 approveFrom err:", err.Error())
				return nil, 0
			}
			txArr = append(txArr, tx)
			nonce += 1
		}
	}
	return txArr, nonce
}

func InitCaseAccount(client *ethclient.Client, txArr []*types.Transaction) []*types.Transaction {
	nonce, err := client.NonceAt(context.Background(), tool.Cfg.DeployAddr, nil)
	if err != nil {
		log.Error("Init account - get acc nonce err:", err)
		return nil
	}
	accArr := tool.ParseAccountConfig("./config/account.json")
	auth, err := bind.NewKeyedTransactorWithChainID(tool.Cfg.DeployPrv, tool.Cfg.ChainId)
	auth.NoSend = true
	amount := big.NewInt(0)
	amount.SetString("10000000000000000000", 10)
	trade := &coin.Trade{
		FromPrv:   tool.Cfg.DeployPrv,
		FromNonce: big.NewInt(int64(nonce)),
		Amount:    amount,
		Gas:       big.NewInt(50000),
		GasPrice:  big.NewInt(1000000000),
		ChainId:   tool.Cfg.ChainId,
	}
	contractMap := tool.ParseContractConfig("./config/contractConfig.json")
	contractLen := len(contractMap)
	//初始化From账户-500tx
	if tool.Cfg.IsInitAccCoin {
		tempTxArr := make([]*types.Transaction, 0)
		tempTxArr, nonce = InitAccountCoin(tempTxArr, nonce, amount, accArr)
		txArr = append(txArr, tempTxArr...)
	}
	//初始化转账合约账户余额-10tx
	if tool.Cfg.IsInitContractCoin {
		tempTxArr := make([]*types.Transaction, 0)
		tempTxArr, nonce = InitContractCoin(tempTxArr, nonce, contractLen, contractMap, trade)
		txArr = append(txArr, tempTxArr...)
	}
	// 初始化erc20 并且mint
	if tool.Cfg.IsInitErc20 {
		tempTxArr := make([]*types.Transaction, 0)
		tempTxArr, nonce = InitErc20Acc(tempTxArr, nonce, contractLen, contractMap, accArr, client, auth)
		txArr = append(txArr, tempTxArr...)
	}
	//初始化erc721 并且mint
	if tool.Cfg.IsInitErc721Mint {
		tempTxArr := make([]*types.Transaction, 0)
		tempTxArr, nonce = InitErc721AccMint(tempTxArr, nonce, contractLen, contractMap, accArr, client, auth)
		txArr = append(txArr, tempTxArr...)
	}
	if tool.Cfg.IsInitErc721 {
		tempTxArr := make([]*types.Transaction, 0)
		tempTxArr, nonce = InitErc721AccTransfer(tempTxArr, nonce, contractLen, contractMap, accArr, client, auth)
		txArr = append(txArr, tempTxArr...)
	}
	log.Info("Init balance successful,wait send tran...........")
	return txArr
}

func GetTranFunc(caseType int, startIndex int, txCount int, nonceMap *sync.Map, accArr *tool.AccArrFormat, client *ethclient.Client, contractArr []map[string]common.Address) ([]*types.Transaction, func() bool) {
	txArr := make([]*types.Transaction, 0)
	arr := GetAccArr(accArr, startIndex, txCount)
	switch caseType {
	// 不相关-普通转账交易
	case 1:
		{
			return NrCoinTranCase(txCount, arr, nonceMap, txArr, client)
		}
	// 不相关-单层-调用合约转账
	case 2:
		{
			return NrContractCoinTranCase(txCount, arr, contractArr[0], nonceMap, txArr, client)
		}
	// 不相关-多层-调用合约转账
	case 3:
		{
			return NrContractOneCoinTranCase(txCount, arr, contractArr[1], nonceMap, txArr, client)
		}
	// 不相关-单层-写变量
	case 4:
		{
			return NrWRTranCase(arr, contractArr[0], nonceMap, txArr, client)
		}
	// 不相关-多层-写变量
	case 5:
		{
			return NrOneWRTranCase(arr, contractArr[1], nonceMap, txArr, client)
		}
	// 不相关-单层-Erc20转账
	case 6:
		{
			return NrErc20TranCase(txCount, arr, contractArr[2], nonceMap, txArr, client)
		}
	// 不相关-多层-Erc20转账
	case 7:
		{
			return NrOneErc20TranCase(txCount, arr, contractArr[0], nonceMap, txArr, client)
		}
	// 不相关-单层-Erc721转账
	case 8:
		{
			return NrErc721TranCase(startIndex, txCount, arr, contractArr[1], nonceMap, txArr, client)
		}
	// 不相关-多层-Erc721转账
	case 9:
		{
			return NrOneErc721TranCase(startIndex, txCount, arr, contractArr[2], nonceMap, txArr, client)
		}
	// 部分相关-普通转账-接受者相关
	case 10:
		{
			return PrReCoinTranCase(txCount, arr, nonceMap, txArr, client)
		}
	// 部分相关-普通转账-发送者相关
	case 11:
		{
			return PrSpCoinTranCase(txCount, arr, nonceMap, txArr, client)
		}
	// 部分相关-普通转账-发送者和接受者部分相关
	case 12:
		{
			return PrSpReCoinTranCase(txCount, arr, nonceMap, txArr, client)
		}
	// 部分相关-普通转账-发送者和接受者全部相关
	case 13:
		{
			return AllPrSpReCoinTranCase(txCount, arr, nonceMap, txArr, client)
		}
	// 部分相关-合约转账-接受者相关
	case 14:
		{
			return PrReContractCoinTranCase(txCount, arr, contractArr, nonceMap, txArr, client)
		}
	// 部分相关-合约转账-发送者相关-不同账户调用同一个合约给不同的账户进行转账
	case 15:
		{
			return PrSpContractCoinTranCase(txCount, arr, contractArr[3], nonceMap, txArr, client)
		}
	// 部分相关-合约转账-发送者接受者相关
	case 16:
		{
			return PrReSpContractCoinTranCase(txCount, arr, contractArr, nonceMap, txArr, client)
		}
	// 部分相关-多层-合约转账-接受者相关
	case 17:
		{
			return PrReContractOneCoinTranCase(txCount, arr, contractArr, nonceMap, txArr, client)
		}
	// 部分相关-多层-合约转账-发送者相关
	case 18:
		{
			return PrSpContractOneCoinTranCase(txCount, arr, contractArr, nonceMap, txArr, client)
		}
	// 部分相关-单层-写变量-合约接受者相关
	case 19:
		{
			return PrReWRTranCase(arr, contractArr[0], nonceMap, txArr, client)
		}
	// 部分相关-单层-写变量-合约发送者相关
	case 20:
		{
			return PrSpWRTranCase(arr, contractArr[0], nonceMap, txArr, client)
		}
	// 部分相关-多层-写变量-合约接受者相关
	case 21:
		{
			return PrReOneWRTranCase(arr, contractArr, nonceMap, txArr, client)
		}
	// 部分相关-单层-erc20-向多个接受者进行转账
	case 22:
		{
			return PrReErc20TranCase(txCount, arr, contractArr[0], nonceMap, txArr, client)
		}
	// 部分相关-多层-erc20-transfer-向多个接受者进行转账
	case 23:
		{
			return PrSpOneErc20TranCase(txCount, arr, contractArr[1], nonceMap, txArr, client)
		}
	// 部分相关-单层-erc20-transferFrom-A->B 转账
	case 24:
		{
			return PrReSpErc20TranCase(txCount, arr, contractArr[2], nonceMap, txArr, client)
		}
	// 部分相关-多层-erc20-transferFrom-发送者相关
	case 25:
		{
			return PrReOneErc20TranCase(txCount, arr, contractArr[3], nonceMap, txArr, client)
		}
		// 部分相关-多层-erc20-transferFrom-发送者相关
	case 30:
		{
			return PrReOneErc20TranCase1(txCount, arr, contractArr[6], nonceMap, txArr, client)
		}
	// Erc20 approve
	case 26:
		{
			return PrSpErc20ApproveTranCase(txCount, arr, contractArr[4], nonceMap, txArr, client)
		}
	// Erc721 接受者相关
	case 27:
		{
			return PrReErc721TranCase(startIndex, txCount, arr, contractArr[5], nonceMap, txArr, client)
		}
	//多层-合约转账- 先set合约的对象地址-〉调用合约的方法 ：两笔交易并行发送，同一个块中执行
	case 28:
		{
			return PrReContractOneCoinTranTwiceCase(txCount, arr, contractArr, nonceMap, txArr, client)
		}

	// Erc721 发送者相关
	case 29:
		{
			return PrReErc721TranFromCase(startIndex, txCount, arr, contractArr[5], nonceMap, txArr, client)
		}
	default:
		{
			return nil, nil
		}
	}
	return nil, nil
}

func GetAccArr(arr *tool.AccArrFormat, startIndex int, count int) *tool.AccArrFormat {
	fromAddr := make([]common.Address, 0)
	fromPrv := make([]*ecdsa.PrivateKey, 0)
	toAddr := make([]common.Address, 0)
	toPrv := make([]*ecdsa.PrivateKey, 0)

	for i := startIndex; i < startIndex+count; i++ {
		fromAddr = append(fromAddr, arr.FromAddr[i])
		fromPrv = append(fromPrv, arr.FromPrv[i])
		toAddr = append(toAddr, arr.ToAddr[i])
		toPrv = append(toPrv, arr.ToPrv[i])
	}
	accStruct := &tool.AccArrFormat{
		FromAddr: fromAddr,
		FromPrv:  fromPrv,
		ToAddr:   toAddr,
		ToPrv:    toPrv,
	}
	return accStruct
}

func SendTransaction(trans []*types.Transaction) {
	client := tool.Cfg.RpcNode
	//for i := 0; i < 8; i++ {
	//	go func() {
	//		for {
	//			select {
	//			case tran := <-tranChanel:
	//				err := client.SendTransaction(context.Background(), tran)
	//				log.Info("Send tran:", tran.Hash(), ",nonce:", tran.Nonce())
	//				if err != nil {
	//					log.Error("send tx error:", err.Error(), ",tx:", tran.Hash().Hex())
	//					return
	//				}
	//			}
	//		}
	//	}()
	//}
	for i := 0; i < len(trans); i++ {
		tran := trans[i]
		//tranChanel <- tran
		err := client.SendTransaction(context.Background(), tran)
		log.Info("Send tran:", tran.Hash(), ",nonce:", tran.Nonce())
		if err != nil {
			log.Error("send tx error:", err.Error(), ",tx:", tran.Hash().Hex())
			return
		}
		log.Info("Input count:", i)
	}
	time.Sleep(time.Second * time.Duration(tool.Cfg.SendTranAfterSleep))
	log.Info("Send tran successful")
}
