package operator

import (
	"contract-case/constant"
	"contract-case/contract/coin"
	contractErc20 "contract-case/contract/erc20"
	contractErc721 "contract-case/contract/erc721"
	readWrite "contract-case/contract/variable_state"
	contractCoin "contract-case/contract_abi/coin_transfer/compile"
	erc20 "contract-case/contract_abi/erc20_transfer/compile"
	erc721 "contract-case/contract_abi/erc721_transfer/compile"
	contractReadWrite "contract-case/contract_abi/read_write/compile"
	"contract-case/log"
	"contract-case/tool"
	"contract-case/util"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"sync"
)

// NrCoinTranCase 普通转账
func NrCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTrade := &coin.Trade{
		Amount:   amount,
		Gas:      big.NewInt(21000),
		GasPrice: big.NewInt(10000000000),
		ChainId:  tool.Cfg.ChainId,
	}
	sendBeforeBal := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		toAddr := testAccArr.ToAddr[i]
		fromAddr := testAccArr.FromAddr[i]
		beforeBal := util.GetAccBal(client, fromAddr)
		if beforeBal == nil {
			return nil, nil
		}
		sendBeforeBal[fromAddr] = beforeBal
		coinTrade.FromPrv = testAccArr.FromPrv[i]
		value, _ := accountNonceMap.Load(fromAddr)
		coinTrade.FromNonce = big.NewInt(value.(int64))
		coinTrade.ToAddr = &toAddr
		tx, err := coinTrade.CoinTransferSignTx()
		if err != nil {
			log.Error("constructor coin transfer tx err:", err.Error())
			continue
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			fromAddr := testAccArr.FromAddr[i]
			sendAfterBal := util.GetAccBal(client, fromAddr)
			if sendAfterBal == nil {
				log.Error("Verify error:", "get sendAfterBal err")
				return false
			}
			subBal := sendBeforeBal[fromAddr].Sub(sendBeforeBal[fromAddr], sendAfterBal)
			if subBal.Cmp(amount) != 1 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// NrContractCoinTranCase 单层-调用合约转账
func NrContractCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTranAddr := contractMap[constant.COIN_TRANSFER]
	contractCoinTrade := &coin.ContractCoinTrade{
		Amount: amount,
	}
	coinTransfer, err := contractCoin.NewChainCoinTransfer(coinTranAddr, client)
	if err != nil {
		log.Error("NewChainCoinTransfer error:", err.Error())
		return nil, nil
	}
	sendBeforeBal := util.GetAccBal(client, coinTranAddr)
	if sendBeforeBal == nil {
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		contractCoinTrade.ToAddr = []common.Address{testAccArr.ToAddr[i]}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(1000000000)
		tx, err := contractCoinTrade.TransferSignTx(auth, coinTransfer)
		if err != nil {
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		sendAfterBal := util.GetAccBal(client, coinTranAddr)
		if sendAfterBal == nil {
			return false
		}
		if sendBeforeBal.Sub(sendBeforeBal, sendAfterBal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// NrContractOneCoinTranCase 多层-调用合约转账
func NrContractOneCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	coinTranAddr := contractMap[constant.COIN_TRANSFER]
	amount := big.NewInt(100000000000000000)
	contractCoinOneTrade := &coin.ContractCoinOneTrade{
		ChainContractAdr: contractMap[constant.COIN_TRANSFER],
		Amount:           amount,
	}
	oneChainCoinTransfer, err := contractCoin.NewOneChainCoinTransfer(contractMap[constant.ONE_COIN_TRANSFER], client)
	if err != nil {
		log.Error("NewOneChainCoinTransfer error:", err.Error())
		return nil, nil
	}
	sendBeforeBal := util.GetAccBal(client, coinTranAddr)
	if sendBeforeBal == nil {
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		contractCoinOneTrade.ToAddr = []common.Address{testAccArr.ToAddr[i]}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := contractCoinOneTrade.OneTransferSignTx(auth, oneChainCoinTransfer)
		if err != nil {
			log.Error("OneTransferSignTx error:", err.Error())
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		sendAfterBal := util.GetAccBal(client, coinTranAddr)
		if sendAfterBal == nil {
			return false
		}
		if sendBeforeBal.Sub(sendBeforeBal, sendAfterBal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// NrWRTranCase 单层-写变量
func NrWRTranCase(testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	readWriteTrade := &readWrite.ReadWriteTrade{
		KeyParam:       big.NewInt(1),
		ValueParam:     big.NewInt(1),
		StringParam:    "test",
		UintParam:      big.NewInt(1),
		ValueArrParam:  []*big.Int{big.NewInt(1), big.NewInt(2)},
		StringArrParam: []string{"1", "2"},
	}
	readWriteVar, err := contractReadWrite.NewReadWriteVar(contractMap[constant.READ_WRITE], client)
	if err != nil {
		log.Error("NewReadWriteVar error:", err.Error())
		return nil, nil
	}
	funcStr := []string{
		"PushVarMapArray",
		"PushVarStringArray",
		"PushVarUintArray",
		"SetVarString",
		"SetVarMap",
		"SetVarMapArray",
		"SetVarUint256",
		"SetVarStringArray",
		"SetVarUintArray",
	}
	for i := 0; i < len(funcStr); i++ {
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		tx := util.ReadWriteCase(testAccArr.FromPrv[i], funcStr[i], big.NewInt(value.(int64)), readWriteTrade, readWriteVar)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		varString, err := readWriteVar.GetVarString(&bind.CallOpts{})
		if err != nil {
			log.Error("GetVarString error:", err.Error())
			return false
		}
		if varString != readWriteTrade.StringParam {
			log.Error(varString, " != ", readWriteTrade.StringParam)
			return false
		}
		return true
	}
	return resTxArr, verify
}

// NrOneWRTranCase 多层-写变量
func NrOneWRTranCase(testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	readWriteAddr := contractMap[constant.READ_WRITE]
	readWriteTrade := &readWrite.OneReadWriteTrade{
		ReadWriteContract: readWriteAddr,
		KeyParam:          big.NewInt(1),
		ValueParam:        big.NewInt(1),
		StringParam:       "test",
		UintParam:         big.NewInt(1),
		ValueArrParam:     []*big.Int{big.NewInt(1), big.NewInt(2)},
		StringArrParam:    []string{"1", "2"},
	}
	readWriteVar, err := contractReadWrite.NewOneReadWriteVar(contractMap[constant.ONE_READ_WRITE], client)
	if err != nil {
		log.Error("NewReadWriteVar error:", err.Error())
		return nil, nil
	}
	funcStr := []string{
		"PushVarMapArray",
		"PushVarStringArray",
		"PushVarUintArray",
		"SetVarString",
		"SetVarMap",
		"SetVarMapArray",
		"SetVarUint256",
		"SetVarStringArray",
		"SetVarUintArray",
	}
	for i := 0; i < len(funcStr); i++ {
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		tx := util.OneReadWriteCase(testAccArr.FromPrv[i], funcStr[i], big.NewInt(value.(int64)), readWriteTrade, *readWriteVar)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		varString, err := readWriteVar.OneGetVarString(&bind.CallOpts{}, readWriteAddr)
		if err != nil {
			log.Error("GetVarString error:", err.Error())
			return false
		}
		if varString != readWriteTrade.StringParam {
			log.Error(varString, " != ", readWriteTrade.StringParam)
			return false
		}
		return true
	}
	return resTxArr, verify
}

// NrErc20TranCase 单层-Erc20转账
func NrErc20TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	erc20Addr := contractMap[constant.ERC20]
	erc20Trade := &contractErc20.Erc20Trade{
		Erc20Addr: erc20Addr,
		Amount:    big.NewInt(10000000000000000),
	}
	erc20Contract, err := erc20.NewERC20(erc20Addr, client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	tokenTransfer, err := erc20.NewTokenTransfer(contractMap[constant.TOKEN20_TRANSFER], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	beforeToAddrErc20Bal := make(map[common.Address]*big.Int, 0)
	beforeToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
	if err != nil {
		log.Error("Get token20ContractBal erc20 balanceOf error:", err.Error())
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		auth.NoSend = true
		erc20Trade.ToAddr = []common.Address{testAccArr.ToAddr[i]}
		toBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeToAddrErc20Bal[testAccArr.ToAddr[i]] = toBal
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		tx, err := erc20Trade.Erc20TransferSignTx(auth, tokenTransfer)
		if err != nil {
			log.Error("Erc20TransferSignTx error:", err.Error())
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
		erc20Trade.FromAddr = testAccArr.FromAddr[i]
		auth.Nonce = auth.Nonce.Add(auth.Nonce, big.NewInt(1))
		tx, err = erc20Trade.Erc20TransferFromSignTx(auth, tokenTransfer)
		if err != nil {
			log.Error("Erc20TransferFromSignTx error:", err.Error())
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		afterToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return false
		}
		if beforeToken20Bal.Sub(beforeToken20Bal, afterToken20Bal).Cmp(big.NewInt(1).Mul(big.NewInt(10000000000000000), big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		for i := 0; i < txCount; i++ {
			of, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if of.Sub(of, beforeToAddrErc20Bal[testAccArr.ToAddr[i]]).Cmp(big.NewInt(1).Mul(big.NewInt(10000000000000000), big.NewInt(2))) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// NrOneErc20TranCase 多层-Erc20转账
func NrOneErc20TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(1000000000000)
	oneErc20Trade := &contractErc20.OneErc20Trade{
		TokenTransferAddr: contractMap[constant.TOKEN20_TRANSFER],
		Erc20Addr:         contractMap[constant.ERC20],
		Amount:            amount,
	}
	tokenTransfer, err := erc20.NewOneTokenTransfer(contractMap[constant.ONE_TOKEN20_TRANSFER], client)
	if err != nil {
		return nil, nil
	}
	erc20Contract, err := erc20.NewERC20(contractMap[constant.ERC20], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	beforeToAddrErc20Bal := make(map[common.Address]*big.Int, 0)
	beforeToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
	log.Info("beforeToken20Bal:", beforeToken20Bal.String())
	if err != nil {
		log.Error("Get token20ContractBal erc20 balanceOf error:", err.Error())
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		oneErc20Trade.ToAddr = []common.Address{testAccArr.ToAddr[i]}
		toBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeToAddrErc20Bal[testAccArr.ToAddr[i]] = toBal
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := oneErc20Trade.Erc20OneTransferTx(auth, tokenTransfer)
		if err != nil {
			return nil, nil
		}
		log.Info("Erc20OneTransferTx tx:", tx.Hash().Hex())
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		afterToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
		if err != nil {
			log.Error("Get token20ContractBal erc20 balanceOf error:", err.Error())
			return false
		}
		for i := 0; i < txCount; i++ {
			of, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if of.Sub(of, beforeToAddrErc20Bal[testAccArr.ToAddr[i]]).Cmp(amount) != 0 {
				return false
			}
		}
		if beforeToken20Bal.Sub(beforeToken20Bal, afterToken20Bal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// NrErc721TranCase 单层-Erc721转账
func NrErc721TranCase(startIndex int, txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	erc721Trade := &contractErc721.Erc721Trade{
		Erc721Addr: contractMap[constant.ERC721],
	}
	erc721Contract, err := erc721.NewERC721(contractMap[constant.ERC721], client)
	if err != nil {
		return nil, nil
	}
	tokenTransfer, err := erc721.NewNftTokenTransfer(contractMap[constant.TOKEN721_TRANSFER], client)
	if err != nil {
		return nil, nil
	}
	beforeToErc721Bal := make(map[common.Address]*big.Int, 0)
	beforeFromErc721Bal := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
			continue
		}
		erc721Trade.ToAddr = testAccArr.ToAddr[i]
		toErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc721 bal error:", err.Error())
			return nil, nil
		}
		fromErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
		if err != nil {
			log.Error("Get erc721 bal error:", err.Error())
			return nil, nil
		}
		beforeToErc721Bal[testAccArr.ToAddr[i]] = toErc721Bal
		beforeFromErc721Bal[testAccArr.FromAddr[i]] = fromErc721Bal
		erc721Trade.FromAddr = testAccArr.FromAddr[i]
		erc721Trade.TokenId = big.NewInt(int64(startIndex))
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := erc721Trade.Erc721TransferFromSignTx(auth, tokenTransfer)
		if err != nil {
			log.Error("Erc721TransferFromSignTx error:", err.Error())
			return nil, nil
		}
		log.Info("Erc721TransferFromSignTx tx:", tx.Hash().Hex())
		resTxArr = append(resTxArr, tx)
		startIndex++
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			toErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc721 bal error:", err.Error())
				return false
			}
			if toErc721Bal.Sub(toErc721Bal, beforeToErc721Bal[testAccArr.ToAddr[i]]).Cmp(big.NewInt(1)) != 0 {
				return false
			}
			fromErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
			if err != nil {
				log.Error("Get erc721 bal error:", err.Error())
				return false
			}
			if beforeFromErc721Bal[testAccArr.FromAddr[i]].Sub(beforeFromErc721Bal[testAccArr.FromAddr[i]], fromErc721Bal).Cmp(big.NewInt(1)) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// NrOneErc721TranCase 多层-Erc721转账
func NrOneErc721TranCase(startIndex int, txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	oneErc721Trade := &contractErc721.OneErc721Trade{
		NftTokenTransferAddr: contractMap[constant.TOKEN721_TRANSFER],
		Erc721Addr:           contractMap[constant.ERC721],
	}
	erc721Contract, err := erc721.NewERC721(contractMap[constant.ERC721], client)
	if err != nil {
		return nil, nil
	}
	tokenTransfer, err := erc721.NewOneNftTokenTransfer(contractMap[constant.ONE_TOKEN721_TRANSFER], client)
	if err != nil {
		return nil, nil
	}
	beforeToErc721Bal := make(map[common.Address]*big.Int, 0)
	beforeFromErc721Bal := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		oneErc721Trade.ToAddr = testAccArr.ToAddr[i]
		oneErc721Trade.FromAddr = testAccArr.FromAddr[i]
		toErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc721 bal error:", err.Error())
			return nil, nil
		}
		fromErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
		if err != nil {
			log.Error("Get erc721 bal error:", err.Error())
			return nil, nil
		}
		beforeToErc721Bal[testAccArr.ToAddr[i]] = toErc721Bal
		beforeFromErc721Bal[testAccArr.FromAddr[i]] = fromErc721Bal
		oneErc721Trade.TokenId = big.NewInt(int64(startIndex))
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := oneErc721Trade.Erc721OneTransferFromSignTx(auth, tokenTransfer)
		if err != nil {
			log.Error("Erc721OneTransferFromSignTx error:", err.Error())
			return nil, nil
		}
		log.Info("Erc721OneTransferFromSignTx tx:", tx.Hash().Hex())
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			toErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc721 bal error:", err.Error())
				return false
			}
			if toErc721Bal.Sub(toErc721Bal, beforeToErc721Bal[testAccArr.ToAddr[i]]).Cmp(big.NewInt(1)) != 0 {
				return false
			}
			fromErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
			if err != nil {
				log.Error("Get erc721 bal error:", err.Error())
				return false
			}
			if beforeFromErc721Bal[testAccArr.FromAddr[i]].Sub(beforeFromErc721Bal[testAccArr.FromAddr[i]], fromErc721Bal).Cmp(big.NewInt(1)) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}
