package operator

import (
	"context"
	"contract-case/constant"
	"contract-case/contract/coin"
	contractErc20 "contract-case/contract/erc20"
	contractErc721 "contract-case/contract/erc721"
	readWrite "contract-case/contract/variable_state"
	contractCoin "contract-case/contract_abi/coin_transfer/compile"
	erc20 "contract-case/contract_abi/erc20_transfer/compile"
	erc721 "contract-case/contract_abi/erc721_transfer/compile"
	contractReadWrite "contract-case/contract_abi/read_write/compile"
	"contract-case/log"
	"contract-case/tool"
	"contract-case/util"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"sync"
)

// PrReCoinTranCase 普通转账-接受者相关
func PrReCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTrade := &coin.Trade{
		Amount:   amount,
		Gas:      big.NewInt(21000),
		GasPrice: big.NewInt(10000000000),
		ChainId:  tool.Cfg.ChainId,
	}
	sendBeforeBal := make(map[common.Address]*big.Int, 0)
	toAddr := testAccArr.ToAddr[0]
	beforeToBal := util.GetAccBal(client, toAddr)
	if beforeToBal == nil {
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		fromAddr := testAccArr.FromAddr[i]
		beforeBal := util.GetAccBal(client, testAccArr.FromAddr[i])
		if beforeBal == nil {
			return nil, nil
		}
		sendBeforeBal[fromAddr] = beforeBal
		coinTrade.FromPrv = testAccArr.FromPrv[i]
		value, _ := accountNonceMap.Load(fromAddr)
		coinTrade.FromNonce = big.NewInt(value.(int64))
		coinTrade.ToAddr = &toAddr
		tx, err := coinTrade.CoinTransferSignTx()
		if err != nil {
			log.Error("constructor coin transfer tx err:", err.Error())
			continue
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			fromAddr := testAccArr.FromAddr[i]
			sendAfterBal := util.GetAccBal(client, fromAddr)
			if sendAfterBal == nil {
				return false
			}
			if sendBeforeBal[fromAddr].Sub(sendBeforeBal[fromAddr], sendAfterBal).Cmp(amount) != 1 {
				return false
			}
		}
		reToAfterBal := util.GetAccBal(client, toAddr)
		if reToAfterBal == nil {
			return false
		}
		if reToAfterBal.Sub(reToAfterBal, beforeToBal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrSpCoinTranCase 普通转账-发送者相关
func PrSpCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTrade := &coin.Trade{
		Amount:   amount,
		Gas:      big.NewInt(21000),
		GasPrice: big.NewInt(10000000000),
		ChainId:  tool.Cfg.ChainId,
	}
	reBeforeBal := make(map[common.Address]*big.Int, 0)
	fromAddr := testAccArr.FromAddr[0]
	coinTrade.FromPrv = testAccArr.FromPrv[0]
	value, _ := accountNonceMap.Load(fromAddr)
	coinTrade.FromNonce = big.NewInt(value.(int64))
	beforeFromBal := util.GetAccBal(client, fromAddr)
	if beforeFromBal == nil {
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		toAddr := testAccArr.ToAddr[i]
		beforeBal := util.GetAccBal(client, toAddr)
		if beforeBal == nil {
			return nil, nil
		}
		reBeforeBal[toAddr] = beforeBal
		coinTrade.ToAddr = &toAddr
		tx, err := coinTrade.CoinTransferSignTx()
		if err != nil {
			log.Error("constructor coin transfer tx err:", err.Error())
			continue
		}
		coinTrade.FromNonce = big.NewInt(value.(int64) + 1)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			toAddr := testAccArr.ToAddr[i]
			reAfterBal := util.GetAccBal(client, toAddr)
			if reAfterBal == nil {
				return false
			}
			if reAfterBal.Sub(reAfterBal, reBeforeBal[toAddr]).Cmp(amount) != 0 {
				return false
			}
		}
		sendAfterBal := util.GetAccBal(client, fromAddr)
		if sendAfterBal == nil {
			return false
		}
		if beforeFromBal.Sub(beforeFromBal, sendAfterBal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 1 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrSpReCoinTranCase 普通转账-发送者和接受者部分相关
func PrSpReCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTrade := &coin.Trade{
		Amount:   amount,
		Gas:      big.NewInt(21000),
		GasPrice: big.NewInt(10000000000),
		ChainId:  tool.Cfg.ChainId,
	}
	// A->B 账户tx
	fromAddr := testAccArr.FromAddr[0]
	coinTrade.FromPrv = testAccArr.FromPrv[0]
	value, _ := accountNonceMap.Load(fromAddr)
	coinTrade.FromNonce = big.NewInt(value.(int64))
	beforeFromBal := util.GetAccBal(client, fromAddr)
	if beforeFromBal == nil {
		return nil, nil
	}
	toAddr := testAccArr.ToAddr[0]
	beforeToBal := util.GetAccBal(client, toAddr)
	if beforeToBal == nil {
		return nil, nil
	}
	coinTrade.ToAddr = &toAddr
	tx, err := coinTrade.CoinTransferSignTx()
	if err != nil {
		log.Error("constructor coin transfer tx err:", err.Error())
	}
	resTxArr = append(resTxArr, tx)

	// B->C 账户tx
	toAddr1 := testAccArr.ToAddr[1]
	beforeTo1Bal := util.GetAccBal(client, toAddr1)
	if beforeTo1Bal == nil {
		return nil, nil
	}
	toNonce, _ := accountNonceMap.Load(toAddr)
	coinTrade.FromPrv = testAccArr.ToPrv[0]
	coinTrade.FromNonce = big.NewInt(toNonce.(int64))
	coinTrade.ToAddr = &toAddr1
	tx, err = coinTrade.CoinTransferSignTx()
	if err != nil {
		log.Error("constructor coin transfer tx err:", err.Error())
	}
	resTxArr = append(resTxArr, tx)

	verify := func() bool {
		// A账户余额校验
		sendFromAfterBal := util.GetAccBal(client, fromAddr)
		if sendFromAfterBal == nil {
			return false
		}
		if beforeFromBal.Sub(beforeFromBal, sendFromAfterBal).Cmp(amount) != 1 {
			return false
		}
		// B账户余额校验
		reToAfterBal := util.GetAccBal(client, toAddr)
		if reToAfterBal == nil {
			return false
		}
		if beforeToBal.Sub(beforeToBal, reToAfterBal).Cmp(big.NewInt(0)) != 1 {
			return false
		}
		// C账户余额校验
		reTo1AfterBal := util.GetAccBal(client, toAddr1)
		if reTo1AfterBal == nil {
			return false
		}
		if reTo1AfterBal.Sub(reTo1AfterBal, beforeTo1Bal).Cmp(amount) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// AllPrSpReCoinTranCase 普通转账-发送者和接受者全部相关
func AllPrSpReCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTrade := &coin.Trade{
		Amount:   amount,
		Gas:      big.NewInt(21000),
		GasPrice: big.NewInt(10000000000),
		ChainId:  tool.Cfg.ChainId,
	}
	// A->B 账户tx
	fromAddr := testAccArr.FromAddr[0]
	coinTrade.FromPrv = testAccArr.FromPrv[0]
	value, _ := accountNonceMap.Load(fromAddr)
	coinTrade.FromNonce = big.NewInt(value.(int64))
	beforeFromBal := util.GetAccBal(client, fromAddr)
	if beforeFromBal == nil {
		return nil, nil
	}
	toAddr := testAccArr.ToAddr[0]
	beforeToBal := util.GetAccBal(client, toAddr)
	if beforeToBal == nil {
		return nil, nil
	}
	coinTrade.ToAddr = &toAddr
	tx, err := coinTrade.CoinTransferSignTx()
	if err != nil {
		log.Error("constructor coin transfer tx err:", err.Error())
	}
	resTxArr = append(resTxArr, tx)

	// B->A 账户tx
	toNonce, _ := accountNonceMap.Load(toAddr)
	coinTrade.FromPrv = testAccArr.ToPrv[0]
	coinTrade.FromNonce = big.NewInt(toNonce.(int64))
	coinTrade.ToAddr = &fromAddr
	tx, err = coinTrade.CoinTransferSignTx()
	if err != nil {
		log.Error("constructor coin transfer tx err:", err.Error())
	}
	resTxArr = append(resTxArr, tx)

	verify := func() bool {
		// A账户余额校验
		sendFromAfterBal := util.GetAccBal(client, fromAddr)
		if sendFromAfterBal == nil {
			return false
		}
		if beforeFromBal.Sub(beforeFromBal, sendFromAfterBal).Cmp(big.NewInt(0)) != 1 {
			return false
		}
		// B账户余额校验
		reToAfterBal := util.GetAccBal(client, toAddr)
		if reToAfterBal == nil {
			return false
		}
		if beforeToBal.Sub(beforeToBal, reToAfterBal).Cmp(big.NewInt(0)) != 1 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrReContractCoinTranCase 合约转账-接受者相关
func PrReContractCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap []map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	contractCoinTrade := &coin.ContractCoinTrade{
		Amount: amount,
	}
	sendBeforeContractBalMap := make(map[common.Address]*big.Int, 0)
	toAddr := testAccArr.ToAddr[0]
	reBeforeToBal := util.GetAccBal(client, toAddr)
	if reBeforeToBal == nil {
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		coinTranAddr := contractMap[i][constant.COIN_TRANSFER]
		sendBeforeBal := util.GetAccBal(client, coinTranAddr)
		if sendBeforeBal == nil {
			return nil, nil
		}
		sendBeforeContractBalMap[coinTranAddr] = sendBeforeBal
		coinTransfer, err := contractCoin.NewChainCoinTransfer(coinTranAddr, client)
		if err != nil {
			log.Error("NewChainCoinTransfer error:", err.Error())
			return nil, nil
		}
		contractCoinTrade.ToAddr = []common.Address{toAddr}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := contractCoinTrade.TransferSignTx(auth, coinTransfer)
		if err != nil {
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			coinAddr := contractMap[i][constant.COIN_TRANSFER]
			sendBeforeBal := sendBeforeContractBalMap[coinAddr]
			sendAfterBal := util.GetAccBal(client, coinAddr)
			if sendBeforeBal == nil {
				return false
			}
			if sendBeforeBal.Sub(sendBeforeBal, sendAfterBal).Cmp(amount) != 0 {
				return false
			}
		}
		reAfterToBal := util.GetAccBal(client, toAddr)
		if reAfterToBal == nil {
			return false
		}
		if reAfterToBal.Sub(reAfterToBal, reBeforeToBal).Cmp(big.NewInt(1).Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrSpContractCoinTranCase 合约转账-发送者相关-不同账户调用同一个合约给不同的账户进行转账
func PrSpContractCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTranAddr := contractMap[constant.COIN_TRANSFER]
	contractCoinTrade := &coin.ContractCoinTrade{
		Amount: amount,
	}
	coinTransfer, err := contractCoin.NewChainCoinTransfer(coinTranAddr, client)
	if err != nil {
		log.Error("NewChainCoinTransfer error:", err.Error())
		return nil, nil
	}
	sendBeforeBal := util.GetAccBal(client, coinTranAddr)
	if sendBeforeBal == nil {
		return nil, nil
	}
	beforeToBalMap := make(map[common.Address]*big.Int)
	for i := 0; i < txCount; i++ {
		fromKey := testAccArr.FromPrv[i]
		fromAddr := testAccArr.FromAddr[i]
		toAddr := testAccArr.ToAddr[i]
		contractCoinTrade.ToAddr = []common.Address{toAddr}
		auth, err := bind.NewKeyedTransactorWithChainID(fromKey, tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		reBeforeToBal := util.GetAccBal(client, toAddr)
		if reBeforeToBal == nil {
			return nil, nil
		}
		beforeToBalMap[toAddr] = reBeforeToBal
		value, _ := accountNonceMap.Load(fromAddr)
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := contractCoinTrade.TransferSignTx(auth, coinTransfer)
		if err != nil {
			log.Error("TransferSignTx error:", err.Error())
			return nil, nil
		}
		auth.Nonce = big.NewInt(value.(int64) + 1)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			toAddr := testAccArr.ToAddr[i]
			reAfterToBal := util.GetAccBal(client, toAddr)
			if reAfterToBal == nil {
				return false
			}
			if reAfterToBal.Sub(reAfterToBal, beforeToBalMap[toAddr]).Cmp(amount) != 0 {
				return false
			}
		}
		sendAfterBal, err := client.BalanceAt(context.Background(), coinTranAddr, nil)
		if err != nil {
			log.Error("Get ", coinTranAddr, " balance error: ", err.Error())
			return false
		}
		if sendBeforeBal.Sub(sendBeforeBal, sendAfterBal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrReSpContractCoinTranCase 合约转账-发送者接受者相关
func PrReSpContractCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap []map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	contractCoinTrade := &coin.ContractCoinTrade{
		Amount: amount,
	}
	coinTranAddr := contractMap[0][constant.COIN_TRANSFER]
	coinTranAddr1 := contractMap[1][constant.COIN_TRANSFER]
	sendBeforeCoinBal := util.GetAccBal(client, coinTranAddr)
	if sendBeforeCoinBal == nil {
		return nil, nil
	}
	sendBeforeCoin1Bal := util.GetAccBal(client, coinTranAddr1)
	if sendBeforeCoin1Bal == nil {
		return nil, nil
	}
	fromKey := testAccArr.FromPrv[0]
	fromAddr := testAccArr.FromAddr[0]
	toAddr := testAccArr.ToAddr[0]
	toPrv := testAccArr.ToPrv[0]
	contractCoinTrade.ToAddr = []common.Address{toAddr}
	reBeforeToBal, err := client.BalanceAt(context.Background(), toAddr, nil)
	if err != nil {
		log.Error("Get ", coinTranAddr, " balance error: ", err.Error())
		return nil, nil
	}
	//  A -> a 合约 -> B 账户转账
	auth, err := bind.NewKeyedTransactorWithChainID(fromKey, tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	value, _ := accountNonceMap.Load(fromAddr)
	auth.Nonce = big.NewInt(value.(int64))
	auth.NoSend = true
	auth.GasLimit = 500000
	auth.GasPrice = big.NewInt(10000000000)
	coinTransfer, err := contractCoin.NewChainCoinTransfer(coinTranAddr, client)
	if err != nil {
		log.Error("NewChainCoinTransfer error:", err.Error())
		return nil, nil
	}
	tx, err := contractCoinTrade.TransferSignTx(auth, coinTransfer)
	if err != nil {
		log.Error("TransferSignTx error:", err.Error())
		return nil, nil
	}
	resTxArr = append(resTxArr, tx)

	//  B -> b 合约 -> C 账户转账
	toAddr1 := testAccArr.ToAddr[1]
	reBeforeTo1Bal, err := client.BalanceAt(context.Background(), toAddr1, nil)
	if err != nil {
		log.Error("Get ", coinTranAddr, " balance error: ", err.Error())
		return nil, nil
	}
	auth, err = bind.NewKeyedTransactorWithChainID(toPrv, tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	value, _ = accountNonceMap.Load(toAddr)
	auth.Nonce = big.NewInt(value.(int64))
	auth.NoSend = true
	auth.GasLimit = 500000
	auth.GasPrice = big.NewInt(10000000000)
	contractCoinTrade.ToAddr = []common.Address{toAddr1}
	coinTransfer1, err := contractCoin.NewChainCoinTransfer(coinTranAddr1, client)
	if err != nil {
		log.Error("NewChainCoinTransfer error:", err.Error())
		return nil, nil
	}
	tx, err = contractCoinTrade.TransferSignTx(auth, coinTransfer1)
	if err != nil {
		log.Error("TransferSignTx error:", err.Error())
		return nil, nil
	}
	resTxArr = append(resTxArr, tx)

	verify := func() bool {
		// 校验 a合约
		sendAfterBal := util.GetAccBal(client, coinTranAddr)
		if sendAfterBal == nil {
			return false
		}
		if sendBeforeCoinBal.Sub(sendBeforeCoinBal, sendAfterBal).Cmp(amount) != 0 {
			return false
		}

		// 校验 b合约
		sendAfterCoin1Bal := util.GetAccBal(client, coinTranAddr1)
		if sendAfterBal == nil {
			return false
		}
		if sendBeforeCoin1Bal.Sub(sendBeforeCoin1Bal, sendAfterCoin1Bal).Cmp(amount) != 0 {
			return false
		}

		// 校验 B 账户
		reAfterToBal := util.GetAccBal(client, toAddr)
		if reAfterToBal == nil {
			return false
		}
		if reAfterToBal.Sub(reAfterToBal, reBeforeToBal).Cmp(amount) != -1 {
			return false
		}

		// 校验 C 账户
		reAfterTo1Bal := util.GetAccBal(client, toAddr1)
		if reAfterTo1Bal == nil {
			return false
		}
		if reAfterTo1Bal.Sub(reAfterTo1Bal, reBeforeTo1Bal).Cmp(amount) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrReContractOneCoinTranCase 多层-合约转账-接受者相关
func PrReContractOneCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap []map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	toAddr := testAccArr.ToAddr[0]
	reBeforeToBal := util.GetAccBal(client, toAddr)
	if reBeforeToBal == nil {
		return nil, nil
	}
	sendBeforeContractBalMap := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		coinTranAddr := contractMap[i][constant.COIN_TRANSFER]
		contractCoinOneTrade := &coin.ContractCoinOneTrade{
			ChainContractAdr: contractMap[i][constant.COIN_TRANSFER],
			Amount:           amount,
		}
		oneChainCoinTransfer, err := contractCoin.NewOneChainCoinTransfer(contractMap[i][constant.ONE_COIN_TRANSFER], client)
		if err != nil {
			log.Error("NewOneChainCoinTransfer error:", err.Error())
			return nil, nil
		}
		sendBeforeBal := util.GetAccBal(client, coinTranAddr)
		if sendBeforeBal == nil {
			return nil, nil
		}
		sendBeforeContractBalMap[contractMap[i][constant.COIN_TRANSFER]] = sendBeforeBal
		contractCoinOneTrade.ToAddr = []common.Address{toAddr}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := contractCoinOneTrade.OneTransferSignTx(auth, oneChainCoinTransfer)
		if err != nil {
			log.Error("OneTransferSignTx error:", err.Error())
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			coinAddr := contractMap[i][constant.COIN_TRANSFER]
			sendBeforeBal := sendBeforeContractBalMap[coinAddr]
			sendAfterBal := util.GetAccBal(client, coinAddr)
			if sendAfterBal == nil {
				return false
			}
			if sendBeforeBal.Sub(sendBeforeBal, sendAfterBal).Cmp(amount) != 0 {
				return false
			}
		}
		reAfterToBal := util.GetAccBal(client, toAddr)
		if reAfterToBal == nil {
			return false
		}
		if reAfterToBal.Sub(reAfterToBal, reBeforeToBal).Cmp(big.NewInt(1).Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrSpContractOneCoinTranCase 多层-合约转账-发送者相关
func PrSpContractOneCoinTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap []map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(100000000000000000)
	coinTranAddr := contractMap[0][constant.COIN_TRANSFER]
	contractCoinOneTrade := &coin.ContractCoinOneTrade{
		ChainContractAdr: contractMap[0][constant.COIN_TRANSFER],
		Amount:           amount,
	}
	oneChainCoinTransfer, err := contractCoin.NewOneChainCoinTransfer(contractMap[0][constant.ONE_COIN_TRANSFER], client)
	if err != nil {
		log.Error("NewOneChainCoinTransfer error:", err.Error())
		return nil, nil
	}
	reBeforeToBalMap := make(map[common.Address]*big.Int, 0)
	sendBeforeBal := util.GetAccBal(client, coinTranAddr)
	if sendBeforeBal == nil {
		return nil, nil
	}
	for i := 0; i < txCount; i++ {
		toAddr := testAccArr.ToAddr[i]
		reBeforeToBal := util.GetAccBal(client, toAddr)
		if reBeforeToBal == nil {
			return nil, nil
		}
		reBeforeToBalMap[toAddr] = reBeforeToBal
		contractCoinOneTrade.ToAddr = []common.Address{toAddr}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := contractCoinOneTrade.OneTransferSignTx(auth, oneChainCoinTransfer)
		if err != nil {
			log.Error("OneTransferSignTx error:", err.Error())
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			reAfterToBal := util.GetAccBal(client, testAccArr.ToAddr[i])
			if reAfterToBal == nil {
				return false
			}
			if reAfterToBal.Sub(reAfterToBal, reBeforeToBalMap[testAccArr.ToAddr[i]]).Cmp(amount) != 0 {
				return false
			}
		}

		coinAddr := contractMap[0][constant.COIN_TRANSFER]
		sendAfterBal := util.GetAccBal(client, coinAddr)
		if sendAfterBal == nil {
			return false
		}
		if sendBeforeBal.Sub(sendBeforeBal, sendAfterBal).Cmp(amount.Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}

		return true
	}
	return resTxArr, verify
}

// PrReWRTranCase 单层-写变量-合约接受者相关
func PrReWRTranCase(testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	readWriteTrade := &readWrite.ReadWriteTrade{
		KeyParam:       big.NewInt(1),
		ValueParam:     big.NewInt(1),
		StringParam:    "test",
		UintParam:      big.NewInt(1),
		ValueArrParam:  []*big.Int{big.NewInt(1), big.NewInt(2)},
		StringArrParam: []string{"1", "2"},
	}
	readWriteVar, err := contractReadWrite.NewReadWriteVar(contractMap[constant.READ_WRITE], client)
	if err != nil {
		log.Error("NewReadWriteVar error:", err.Error())
		return nil, nil
	}
	funcStr := []string{
		"PushVarMapArray",
		"PushVarStringArray",
		"PushVarUintArray",
		"SetVarString",
		"SetVarMap",
		"SetVarMapArray",
		"SetVarUint256",
		"SetVarStringArray",
		"SetVarUintArray",
	}
	for i := 0; i < len(funcStr); i++ {
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		tx := util.ReadWriteCase(testAccArr.FromPrv[i], funcStr[i], big.NewInt(value.(int64)), readWriteTrade, readWriteVar)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		varString, err := readWriteVar.GetVarString(&bind.CallOpts{})
		if err != nil {
			log.Error("GetVarString error:", err.Error())
			return false
		}
		if varString != readWriteTrade.StringParam {
			log.Error(varString, " != ", readWriteTrade.StringParam)
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrSpWRTranCase 单层-写变量-合约发送者相关
func PrSpWRTranCase(testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	readWriteTrade := &readWrite.ReadWriteTrade{
		KeyParam:       big.NewInt(1),
		ValueParam:     big.NewInt(1),
		StringParam:    "test",
		UintParam:      big.NewInt(1),
		ValueArrParam:  []*big.Int{big.NewInt(1), big.NewInt(2)},
		StringArrParam: []string{"1", "2"},
	}
	readWriteVar, err := contractReadWrite.NewReadWriteVar(contractMap[constant.READ_WRITE], client)
	if err != nil {
		log.Error("NewReadWriteVar error:", err.Error())
		return nil, nil
	}
	funcStr := []string{
		"PushVarMapArray",
		"PushVarStringArray",
		"PushVarUintArray",
		"SetVarString",
		"SetVarMap",
		"SetVarMapArray",
		"SetVarUint256",
		"SetVarStringArray",
		"SetVarUintArray",
	}
	for i := 0; i < len(funcStr); i++ {
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[0])
		tx := util.ReadWriteCase(testAccArr.FromPrv[0], funcStr[i], big.NewInt(value.(int64)), readWriteTrade, readWriteVar)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		varString, err := readWriteVar.GetVarString(&bind.CallOpts{})
		if err != nil {
			log.Error("GetVarString error:", err.Error())
			return false
		}
		if varString != readWriteTrade.StringParam {
			log.Error(varString, " != ", readWriteTrade.StringParam)
			return false
		}
		return true
	}
	return resTxArr, verify
}

// PrReOneWRTranCase 多层-写变量-合约接受者相关
func PrReOneWRTranCase(testAccArr *tool.AccArrFormat, contractMap []map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	readWriteAddr := contractMap[0][constant.READ_WRITE]
	readWriteTrade := &readWrite.OneReadWriteTrade{
		ReadWriteContract: readWriteAddr,
		KeyParam:          big.NewInt(1),
		ValueParam:        big.NewInt(1),
		StringParam:       "test",
		UintParam:         big.NewInt(1),
		ValueArrParam:     []*big.Int{big.NewInt(1), big.NewInt(2)},
		StringArrParam:    []string{"1", "2"},
	}

	funcStr := []string{
		"PushVarMapArray",
		"PushVarStringArray",
		"PushVarUintArray",
		"SetVarString",
		"SetVarMap",
		"SetVarMapArray",
		"SetVarUint256",
		"SetVarStringArray",
		"SetVarUintArray",
	}
	for i := 0; i < len(funcStr); i++ {
		readWriteVar, err := contractReadWrite.NewOneReadWriteVar(contractMap[i][constant.ONE_READ_WRITE], client)
		if err != nil {
			log.Error("NewReadWriteVar error:", err.Error())
			return nil, nil
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		tx := util.OneReadWriteCase(testAccArr.FromPrv[i], funcStr[i], big.NewInt(value.(int64)), readWriteTrade, *readWriteVar)
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < len(funcStr); i++ {
			readWriteVar, err := contractReadWrite.NewOneReadWriteVar(contractMap[i][constant.ONE_READ_WRITE], client)
			if err != nil {
				log.Error("NewReadWriteVar error:", err.Error())
				return false
			}
			varString, err := readWriteVar.OneGetVarString(&bind.CallOpts{}, readWriteAddr)
			if err != nil {
				log.Error("GetVarString error:", err.Error())
				return false
			}
			if varString != readWriteTrade.StringParam {
				log.Error(varString, " != ", readWriteTrade.StringParam)
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrReErc20TranCase 单层-erc20-向多个接受者进行转账
func PrReErc20TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	erc20Trade := &contractErc20.Erc20Trade{
		Erc20Addr: contractMap[constant.ERC20],
		Amount:    big.NewInt(10000000000000000),
	}
	erc20Contract, err := erc20.NewERC20(contractMap[constant.ERC20], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	tokenTransfer, err := erc20.NewTokenTransfer(contractMap[constant.TOKEN20_TRANSFER], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	beforeToAddrErc20Bal := make(map[common.Address]*big.Int, 0)
	beforeToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
	if err != nil {
		log.Error("Get token20ContractBal erc20 balanceOf error:", err.Error())
		return nil, nil
	}
	toAddrArr := make([]common.Address, 0)
	for i := 0; i < txCount; i++ {
		toAddrArr = append(toAddrArr, testAccArr.ToAddr[i])
		toBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeToAddrErc20Bal[testAccArr.ToAddr[i]] = toBal
	}
	erc20Trade.ToAddr = toAddrArr
	auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[0], tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	auth.NoSend = true
	auth.GasLimit = 500000
	auth.GasPrice = big.NewInt(10000000000)
	value, _ := accountNonceMap.Load(testAccArr.FromAddr[0])
	auth.Nonce = big.NewInt(value.(int64))
	tx, err := erc20Trade.Erc20TransferSignTx(auth, tokenTransfer)
	if err != nil {
		log.Error("Erc20TransferSignTx error:", err.Error())
		return nil, nil
	}
	resTxArr = append(resTxArr, tx)
	verify := func() bool {
		afterToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return false
		}
		if beforeToken20Bal.Sub(beforeToken20Bal, afterToken20Bal).Cmp(big.NewInt(1).Mul(erc20Trade.Amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		for i := 0; i < txCount; i++ {
			of, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if of.Sub(of, beforeToAddrErc20Bal[testAccArr.ToAddr[i]]).Cmp(erc20Trade.Amount) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrSpOneErc20TranCase 多层-erc20-transfer-向多个接受者进行转账
func PrSpOneErc20TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(1000000000000)
	oneErc20Trade := &contractErc20.OneErc20Trade{
		TokenTransferAddr: contractMap[constant.TOKEN20_TRANSFER],
		Erc20Addr:         contractMap[constant.ERC20],
		Amount:            amount,
	}
	tokenTransfer, err := erc20.NewOneTokenTransfer(contractMap[constant.ONE_TOKEN20_TRANSFER], client)
	if err != nil {
		return nil, nil
	}
	erc20Contract, err := erc20.NewERC20(contractMap[constant.ERC20], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	beforeToAddrErc20Bal := make(map[common.Address]*big.Int, 0)
	beforeToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
	if err != nil {
		log.Error("Get token20ContractBal erc20 balanceOf error:", err.Error())
		return nil, nil
	}
	toAddrArr := make([]common.Address, 0)
	for i := 0; i < txCount; i++ {
		toAddrArr = append(toAddrArr, testAccArr.ToAddr[i])
		toBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeToAddrErc20Bal[testAccArr.ToAddr[i]] = toBal
	}
	oneErc20Trade.ToAddr = toAddrArr
	auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[0], tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	value, _ := accountNonceMap.Load(testAccArr.FromAddr[0])
	auth.Nonce = big.NewInt(value.(int64))
	auth.NoSend = true
	auth.GasLimit = 500000
	auth.GasPrice = big.NewInt(10000000000)
	tx, err := oneErc20Trade.Erc20OneTransferTx(auth, tokenTransfer)
	if err != nil {
		return nil, nil
	}
	log.Info("Erc20OneTransferTx tx:", tx.Hash().Hex())
	resTxArr = append(resTxArr, tx)

	verify := func() bool {
		afterToken20Bal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, contractMap[constant.TOKEN20_TRANSFER])
		if err != nil {
			log.Error("Get token20ContractBal erc20 balanceOf error:", err.Error())
			return false
		}
		if beforeToken20Bal.Sub(beforeToken20Bal, afterToken20Bal).Cmp(big.NewInt(1).Mul(amount, big.NewInt(int64(txCount)))) != 0 {
			return false
		}
		for i := 0; i < txCount; i++ {
			of, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if of.Sub(of, beforeToAddrErc20Bal[testAccArr.ToAddr[i]]).Cmp(amount) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrReSpErc20TranCase 单层-erc20-transferFrom-A->B 转账
func PrReSpErc20TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	erc20Trade := &contractErc20.Erc20Trade{
		Erc20Addr: contractMap[constant.ERC20],
		Amount:    big.NewInt(10000000000000000),
	}
	erc20Contract, err := erc20.NewERC20(contractMap[constant.ERC20], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	tokenTransfer, err := erc20.NewTokenTransfer(contractMap[constant.TOKEN20_TRANSFER], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	beforeAddrErc20Bal := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		toBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeAddrErc20Bal[testAccArr.ToAddr[i]] = toBal
		fromBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeAddrErc20Bal[testAccArr.FromAddr[i]] = fromBal

		erc20Trade.FromAddr = testAccArr.FromAddr[i]
		erc20Trade.ToAddr = []common.Address{testAccArr.ToAddr[i]}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		tx, err := erc20Trade.Erc20TransferFromSignTx(auth, tokenTransfer)
		if err != nil {
			log.Error("Erc20TransferSignTx error:", err.Error())
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			toAddr := testAccArr.ToAddr[i]
			to, err := erc20Contract.BalanceOf(&bind.CallOpts{}, toAddr)
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if to.Sub(to, beforeAddrErc20Bal[toAddr]).Cmp(erc20Trade.Amount) != 0 {
				return false
			}

			fromAddr := testAccArr.FromAddr[i]
			from, err := erc20Contract.BalanceOf(&bind.CallOpts{}, fromAddr)
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if beforeAddrErc20Bal[fromAddr].Sub(beforeAddrErc20Bal[fromAddr], from).Cmp(erc20Trade.Amount) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrReOneErc20TranCase 多层-erc20-transferFrom-发送者相关
func PrReOneErc20TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(1000000000000)
	oneErc20Trade := &contractErc20.OneErc20Trade{
		TokenTransferAddr: contractMap[constant.TOKEN20_TRANSFER],
		Erc20Addr:         contractMap[constant.ERC20],
		Amount:            amount,
	}
	tokenTransfer, err := erc20.NewOneTokenTransfer(contractMap[constant.ONE_TOKEN20_TRANSFER], client)
	if err != nil {
		return nil, nil
	}
	erc20Contract, err := erc20.NewERC20(contractMap[constant.ERC20], client)
	if err != nil {
		log.Error("NewTokenTransfer error:", err.Error())
		return nil, nil
	}
	beforeAddrErc20Bal := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		toBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeAddrErc20Bal[testAccArr.ToAddr[i]] = toBal

		fromBal, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
		if err != nil {
			log.Error("Get erc20 balanceOf error:", err.Error())
			return nil, nil
		}
		beforeAddrErc20Bal[testAccArr.FromAddr[i]] = fromBal

		oneErc20Trade.ToAddr = []common.Address{testAccArr.ToAddr[i]}
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 500000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := oneErc20Trade.Erc20OneTransferTx(auth, tokenTransfer)
		if err != nil {
			return nil, nil
		}
		log.Info("Erc20OneTransferTx tx:", tx.Hash().Hex())
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			to, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if to.Sub(to, beforeAddrErc20Bal[testAccArr.ToAddr[i]]).Cmp(amount) != 0 {
				return false
			}

			from, err := erc20Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if beforeAddrErc20Bal[testAccArr.FromAddr[i]].Sub(beforeAddrErc20Bal[testAccArr.FromAddr[i]], from).Cmp(amount) != 1 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrSpErc20ApproveTranCase 单层-erc20-approve
func PrSpErc20ApproveTranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	amount := big.NewInt(0)
	amount.SetString("100000000000000000000000", 10)
	erc20Addr := contractMap[constant.ERC20]
	erc20Contract, err := erc20.NewERC20(erc20Addr, client)
	if err != nil {
		log.Error("NewERC20 error:", err.Error())
		return nil, nil
	}
	beforeFromToAllowanceArr := make([]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		beforeFromToAllowance, err := erc20Contract.Allowance(&bind.CallOpts{}, testAccArr.FromAddr[i], testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get token20ContractBal erc20 allowance error:", err.Error())
			return nil, nil
		}
		beforeFromToAllowanceArr = append(beforeFromToAllowanceArr, beforeFromToAllowance)
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
		}
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasLimit = 1000000000000
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := erc20Contract.Approve(auth, testAccArr.ToAddr[i], amount)
		if err != nil {
			log.Error("Approve failed:", err)
			return nil, nil
		}
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			of, err := erc20Contract.Allowance(&bind.CallOpts{}, testAccArr.FromAddr[i], testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc20 balanceOf error:", err.Error())
				return false
			}
			if of.Sub(of, beforeFromToAllowanceArr[i]) != amount {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrReSpOneErc20ApproveTranCase 多层-erc20-approve
func PrReSpOneErc20ApproveTranCase() {

}

// PrReErc721TranCase 单层-erc721-发送者相关
func PrReErc721TranCase(txCount int, testAccArr *tool.AccArrFormat, contractMap map[string]common.Address, accountNonceMap *sync.Map, resTxArr []*types.Transaction, client *ethclient.Client) ([]*types.Transaction, func() bool) {
	erc721Trade := &contractErc721.Erc721Trade{
		Erc721Addr: contractMap[constant.ERC721],
	}
	erc721Contract, err := erc721.NewERC721(contractMap[constant.ERC721], client)
	if err != nil {
		return nil, nil
	}
	tokenTransfer, err := erc721.NewNftTokenTransfer(contractMap[constant.TOKEN721_TRANSFER], client)
	if err != nil {
		return nil, nil
	}
	beforeToErc721Bal := make(map[common.Address]*big.Int, 0)
	beforeFromErc721Bal := make(map[common.Address]*big.Int, 0)
	for i := 0; i < txCount; i++ {
		auth, err := bind.NewKeyedTransactorWithChainID(testAccArr.FromPrv[i], tool.Cfg.ChainId)
		if err != nil {
			log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
			continue
		}
		erc721Trade.ToAddr = testAccArr.ToAddr[i]
		toErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
		if err != nil {
			log.Error("Get erc721 bal error:", err.Error())
			return nil, nil
		}
		fromErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
		if err != nil {
			log.Error("Get erc721 bal error:", err.Error())
			return nil, nil
		}
		beforeToErc721Bal[testAccArr.ToAddr[i]] = toErc721Bal
		beforeFromErc721Bal[testAccArr.FromAddr[i]] = fromErc721Bal
		erc721Trade.FromAddr = testAccArr.FromAddr[i]
		// todo 需要初始化时知道sender对应的tokenId
		erc721Trade.TokenId = big.NewInt(int64(i))
		value, _ := accountNonceMap.Load(testAccArr.FromAddr[i])
		auth.Nonce = big.NewInt(value.(int64))
		auth.NoSend = true
		auth.GasPrice = big.NewInt(10000000000)
		tx, err := erc721Trade.Erc721TransferFromSignTx(auth, tokenTransfer)
		if err != nil {
			log.Error("Erc721TransferFromSignTx error:", err.Error())
			return nil, nil
		}
		log.Info("Erc721TransferFromSignTx tx:", tx.Hash().Hex())
		resTxArr = append(resTxArr, tx)
	}
	verify := func() bool {
		for i := 0; i < txCount; i++ {
			toErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.ToAddr[i])
			if err != nil {
				log.Error("Get erc721 bal error:", err.Error())
				return false
			}
			if beforeToErc721Bal[testAccArr.ToAddr[i]].Sub(beforeToErc721Bal[testAccArr.ToAddr[i]], toErc721Bal).Cmp(big.NewInt(1)) != 0 {
				return false
			}
			fromErc721Bal, err := erc721Contract.BalanceOf(&bind.CallOpts{}, testAccArr.FromAddr[i])
			if err != nil {
				log.Error("Get erc721 bal error:", err.Error())
				return false
			}
			if beforeFromErc721Bal[testAccArr.FromAddr[i]].Sub(beforeToErc721Bal[testAccArr.FromAddr[i]], fromErc721Bal).Cmp(big.NewInt(1)) != 0 {
				return false
			}
		}
		return true
	}
	return resTxArr, verify
}

// PrSpErc721TranCase 单层-erc721-接受者相关
func PrSpErc721TranCase() {

}

// PrReSpErc721TranCase 单层-erc721-发送接受者相关
func PrReSpErc721TranCase() {

}
