package util

import (
	"context"
	"contract-case/constant"
	readWrite "contract-case/contract/variable_state"
	erc20 "contract-case/contract_abi/erc20_transfer/compile"
	erc721 "contract-case/contract_abi/erc721_transfer/compile"
	contractReadWrite "contract-case/contract_abi/read_write/compile"
	"contract-case/log"
	"contract-case/tool"
	"crypto/ecdsa"
	"encoding/json"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
	"golang.org/x/crypto/sha3"
	"math/big"
	"os"
	"sync"
	"sync/atomic"
	"time"
)

func CallContractData(method string, params ...[]byte) []byte {
	var data []byte
	hash := sha3.NewLegacyKeccak256()
	hash.Write([]byte(method))
	methodID := hash.Sum(nil)[:4]
	data = append(data, methodID...)
	for _, v := range params {
		data = append(data, common.LeftPadBytes(v, 32)...)
	}
	return data
}

func HexToAddress(addr string) common.Address {
	if addr == "" {
		return common.Address{}
	}
	return common.HexToAddress(addr)
}

func GetTxReceipt(client *ethclient.Client, txArr []*types.Transaction) {
	txChanel := make(chan common.Hash, 1000)
	for i := 0; i < 8; i++ {
		go func() {
			select {
			case hash := <-txChanel:
				receipt, err := client.TransactionReceipt(context.Background(), hash)
				if err != nil {
					log.Error("Get tran receipt err:", err)
					return
				}
				if receipt.Status == 0 {
					log.Error("Receipt:", receipt)
				} else {
					log.Infof("Receipt: status: %d,gasUsed: %d", receipt.Status, receipt.GasUsed)
				}
			}
		}()
	}
	for _, tx := range txArr {
		txChanel <- tx.Hash()
	}
	time.Sleep(time.Second * 10)
}

func GetAccountNonce(client *ethclient.Client) *sync.Map {
	accountArr := tool.ParseAccountConfig("./config/account.json")
	addrChan := make(chan common.Address, 1000)
	resSyncMap := &sync.Map{}
	var handleNonceCount int32
	for i := 0; i < 8; i++ {
		go func() {
			for {
				select {
				case addr := <-addrChan:
					balance, err := client.NonceAt(context.Background(), addr, nil)
					if err != nil {
						log.Errorf("Get acc:%s,nonce err:%s", addr.Hash().Hex(), err)
					}
					resSyncMap.Store(addr, int64(balance))
					atomic.AddInt32(&handleNonceCount, 1)
					//log.Info("handleNonceCount:", handleNonceCount)
				}
			}
		}()
	}
	for i := 0; i < len(accountArr.FromAddr); i++ {
		addrChan <- accountArr.FromAddr[i]
		addrChan <- accountArr.ToAddr[i]
	}
	time.Sleep(time.Second * 30)
	if handleNonceCount == int32(tool.Cfg.GenerateAccCount) {
		log.Info("Init all account nonce successful")
	}
	return resSyncMap
}

func GetAccBal(client *ethclient.Client, addr common.Address) *big.Int {
	mutex := sync.Mutex{}
	mutex.Lock()
	defer mutex.Unlock()
	balance, err := client.BalanceAt(context.Background(), addr, nil)
	if err != nil {
		log.Errorf("Get acc:%s,balance err:%s", addr.Hash().Hex(), err.Error())
		return nil
	}
	log.Info(addr.Hash().Hex(), ",balance:", balance)
	return balance
}

func GetAccountBalance(client *ethclient.Client) {
	accountArr := tool.ParseAccountConfig("./config/account.json")
	b, err := client.BalanceAt(context.Background(), tool.Cfg.DeployAddr, nil)
	log.Info(tool.Cfg.DeployAddr, ",balance:", b, err)
	addrChan := make(chan common.Address, 500)
	var handleBalanceCount int32
	for i := 0; i < 8; i++ {
		go func() {
			for {
				select {
				case addr := <-addrChan:
					GetAccBal(client, addr)
					GetAccountErc20BalanceOf(client, tool.Cfg.ContractMap[0][constant.ERC20], addr)
					GetAccountErc721BalanceOf(client, tool.Cfg.ContractMap[0][constant.ERC721], addr)
					atomic.AddInt32(&handleBalanceCount, 1)
				}
			}
		}()
	}
	for i := 0; i < len(accountArr.FromAddr); i++ {
		addrChan <- accountArr.FromAddr[i]
		addrChan <- accountArr.ToAddr[i]
	}
	time.Sleep(time.Second * 30)
	if handleBalanceCount == int32(tool.Cfg.GenerateAccCount) {
		log.Info("Init all account balance successful")
	}
}

func GetAccountErc20BalanceOf(client *ethclient.Client, erc20Addr common.Address, toAddr common.Address) {
	mutex := sync.Mutex{}
	mutex.Lock()
	defer mutex.Unlock()
	newERC20, err := erc20.NewERC20(erc20Addr, client)
	if err != nil {
		log.Error("Init account - new erc720 err:", err.Error())
		return
	}
	of, err := newERC20.BalanceOf(&bind.CallOpts{}, toAddr)
	if err != nil {
		log.Errorf("Get erc20 balance of failed err:%s", err.Error())
		return
	}
	//of, err = newERC20.Allowance(&bind.CallOpts{}, common.HexToAddress("0xfb1bF24c0C7e7C1B89c4A496aACf01aBF26C0B94"), toAddr)
	//if err != nil {
	//	log.Errorf("Get erc20 balance of failed err:%s", err.Error())
	//	return
	//}
	log.Infof("%s erc20 balance of is: %s ", toAddr.Hash().Hex(), of.String())
}

func GetAccountErc721BalanceOf(client *ethclient.Client, erc721Addr common.Address, toAddr common.Address) {
	mutex := sync.Mutex{}
	mutex.Lock()
	defer mutex.Unlock()
	newERC721, err := erc721.NewERC721(erc721Addr, client)
	if err != nil {
		log.Error("Init account - new erc721 err:", err.Error())
		return
	}
	of, err := newERC721.BalanceOf(&bind.CallOpts{}, toAddr)
	if err != nil {
		log.Errorf("Get erc721 balance of failed err:", err.Error())
		return
	}
	log.Infof("%s erc721 balance of is: %d ", toAddr.Hash().Hex(), of)
}

func GenerateContractMap(contractArr []map[string]common.Address) {
	jsonData, err := json.MarshalIndent(contractArr, "", "  ")
	if err != nil {
		log.Error("JSON encoding contract failed: ", err)
	}
	err = os.WriteFile("./config/contractConfig.json", jsonData, 0644)
	if err != nil {
		fmt.Println("File writing contract failed: ", err)
	}
}

func ReadWriteCase(prv *ecdsa.PrivateKey, funcStr string, nonce *big.Int, readWriteTrade *readWrite.ReadWriteTrade, readWriteVar *contractReadWrite.ReadWriteVar) *types.Transaction {
	auth, err := bind.NewKeyedTransactorWithChainID(prv, tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	auth.Nonce = nonce
	auth.NoSend = true
	auth.GasLimit = 500000
	auth.GasPrice = big.NewInt(1000000000000)
	tx := &types.Transaction{}
	switch funcStr {
	case "PushVarMapArray":
		{
			tx, err = readWriteTrade.ReadWriteSignTxPushVarMapArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "PushVarStringArray":
		{
			tx, err = readWriteTrade.ReadWriteSignTxPushVarStringArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}

	case "PushVarUintArray":
		{
			tx, err = readWriteTrade.ReadWriteSignTxPushVarUintArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarString":
		{
			tx, err = readWriteTrade.ReadWriteSignTxSetVarString(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarMap":
		{
			tx, err = readWriteTrade.ReadWriteSignTxSetVarMap(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarMapArray":
		{
			tx, err = readWriteTrade.ReadWriteSignTxSetVarMapArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}

	case "SetVarUint256":
		{
			tx, err = readWriteTrade.ReadWriteSignTxSetVarUint256(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarStringArray":
		{
			tx, err = readWriteTrade.ReadWriteSignTxSetVarStringArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarUintArray":
		{
			tx, err = readWriteTrade.ReadWriteSignTxSetVarUintArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	}
	return tx
}

func OneReadWriteCase(prv *ecdsa.PrivateKey, funcStr string, nonce *big.Int, readWriteTrade *readWrite.OneReadWriteTrade, readWriteVar contractReadWrite.OneReadWriteVar) *types.Transaction {
	auth, err := bind.NewKeyedTransactorWithChainID(prv, tool.Cfg.ChainId)
	if err != nil {
		log.Errorf("DeployTokenTransfer func newKeyedTransactorWithChainID err:", err)
	}
	auth.Nonce = nonce
	auth.NoSend = true
	auth.GasLimit = 500000
	auth.GasPrice = big.NewInt(1000000000000)
	tx := &types.Transaction{}
	switch funcStr {
	case "PushVarMapArray":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxPushVarMapArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "PushVarStringArray":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxPushVarStringArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}

	case "PushVarUintArray":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxPushVarUintArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarString":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxSetVarString(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarMap":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxSetVarMap(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarMapArray":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxSetVarMapArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}

	case "SetVarUint256":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxSetVarUint256(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarStringArray":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxSetVarStringArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	case "SetVarUintArray":
		{
			tx, err = readWriteTrade.OneReadWriteSignTxPushVarUintArray(auth, readWriteVar)
			if err != nil {
				return nil
			}
		}
	}
	return tx
}
