package tool

import (
	"bufio"
	"contract-case/log"
	"crypto/ecdsa"
	"encoding/json"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	"math/big"
	"os"
	"strings"
)

type ConfigJson struct {
	DeployAddr           string `json:"deployAddr"`
	DeployPrv            string `json:"deployPrv"`
	RpcNode              string `json:"rpcNode"`
	ChainId              int64  `json:"chainId"`
	TransferType         []int  `json:"transferType"`
	GenerateAccCount     int    `json:"generateAccCount"`
	IsDeployContract     bool   `json:"isDeployContract"`
	IsInitAccountBalance bool   `json:"isInitAccountBalance"`
	IsInitAccCoin        bool   `json:"isInitAccCoin"`
	IsInitContractCoin   bool   `json:"isInitContractCoin"`
	IsInitErc20          bool   `json:"isInitErc20"`
	IsInitErc721Mint     bool   `json:"isInitErc721Mint"`
	IsInitErc721         bool   `json:"isInitErc721"`
	AccStartIndex        int    `json:"accStartIndex"`
	SendTranAfterSleep   int    `json:"sendTranAfterSleep"`
	IsGetAccBalance      bool   `json:"isGetAccBalance"`
}

type Config struct {
	DeployAddr           common.Address
	DeployPrv            *ecdsa.PrivateKey
	RpcNode              *ethclient.Client
	ChainId              *big.Int
	TransferType         []int
	GenerateAccCount     int
	IsDeployContract     bool
	IsInitAccountBalance bool
	IsInitAccCoin        bool
	IsInitContractCoin   bool
	IsInitErc20          bool
	IsInitErc721Mint     bool
	IsInitErc721         bool
	ContractMap          []map[string]common.Address
	AccStartIndex        int
	SendTranAfterSleep   int
	IsGetAccBalance      bool
}

type AccArr struct {
	FromAddr []string `json:"from"`
	FromPrv  []string `json:"fromPrv"`
	ToAddr   []string `json:"toAddr"`
	ToPrv    []string `json:"toPrv"`
}

type AccArrFormat struct {
	FromAddr []common.Address
	FromPrv  []*ecdsa.PrivateKey
	ToAddr   []common.Address
	ToPrv    []*ecdsa.PrivateKey
}

var (
	cfgJson *ConfigJson
	Cfg     *Config
)

func init() {
	ParseConfig("./config/app.json")
}

func ParseConfig(path string) {
	file, err := os.Open(path)
	if err != nil {
		panic(err)
	}
	defer func(file *os.File) {
		err := file.Close()
		if err != nil {
			log.Error("read file close failed:", err)
		}
	}(file)
	reader := bufio.NewReader(file)
	decoder := json.NewDecoder(reader)
	if err := decoder.Decode(&cfgJson); err != nil {
		log.Error("Config func decode error:", err)
	}
	deployECDSA, err := crypto.HexToECDSA(cfgJson.DeployPrv)
	if err != nil {
		log.Error("Config func HexToECDSA error:", err)
	}
	dial, err := ethclient.Dial(cfgJson.RpcNode)
	if err != nil {
		log.Error("Config func eth client dial error:", err)
	}
	Cfg = &Config{
		DeployAddr:           common.HexToAddress(cfgJson.DeployAddr),
		DeployPrv:            deployECDSA,
		RpcNode:              dial,
		ChainId:              big.NewInt(cfgJson.ChainId),
		TransferType:         cfgJson.TransferType,
		GenerateAccCount:     cfgJson.GenerateAccCount,
		IsDeployContract:     cfgJson.IsDeployContract,
		IsInitAccountBalance: cfgJson.IsInitAccountBalance,
		IsInitAccCoin:        cfgJson.IsInitAccCoin,
		IsInitContractCoin:   cfgJson.IsInitContractCoin,
		IsInitErc20:          cfgJson.IsInitErc20,
		IsInitErc721Mint:     cfgJson.IsInitErc721Mint,
		IsInitErc721:         cfgJson.IsInitErc721,
		ContractMap:          ParseContractConfig("./config/contractConfig.json"),
		AccStartIndex:        cfgJson.AccStartIndex,
		SendTranAfterSleep:   cfgJson.SendTranAfterSleep,
		IsGetAccBalance:      cfgJson.IsGetAccBalance,
	}
}

func ParseContractConfig(path string) []map[string]common.Address {
	data, err := os.ReadFile(path)
	if err != nil {
		panic(err)
	}
	var receive []map[string]string
	err = json.Unmarshal(data, &receive)
	if err != nil {
		panic(err)
	}
	result := make([]map[string]common.Address, 0)
	for _, contractMap := range receive {
		mapAddress := make(map[string]common.Address, 0)
		for key, val := range contractMap {
			mapAddress[key] = common.HexToAddress(val)
		}
		result = append(result, mapAddress)
	}
	return result
}

func ParseAccountConfig(path string) *AccArrFormat {
	data, err := os.ReadFile(path)
	if err != nil {
		panic(err)
	}
	res := &AccArr{}
	err = json.Unmarshal(data, &res)
	if err != nil {
		panic(err)
	}
	fromAddrArr := make([]common.Address, 0)
	fromPrvArr := make([]*ecdsa.PrivateKey, 0)
	toAddrArr := make([]common.Address, 0)
	toPrvArr := make([]*ecdsa.PrivateKey, 0)
	for i := 0; i < len(res.FromPrv); i++ {
		fromAddr := common.HexToAddress(res.FromAddr[i])
		toAddr := common.HexToAddress(res.ToAddr[i])
		fromLeftPrv := strings.TrimPrefix(res.FromPrv[i], "0x")
		fromPrv, _ := crypto.HexToECDSA(fromLeftPrv)
		toLeftPrv := strings.TrimPrefix(res.ToPrv[i], "0x")
		toPrv, _ := crypto.HexToECDSA(toLeftPrv)
		fromAddrArr = append(fromAddrArr, fromAddr)
		fromPrvArr = append(fromPrvArr, fromPrv)
		toAddrArr = append(toAddrArr, toAddr)
		toPrvArr = append(toPrvArr, toPrv)
	}
	accArrFormat := &AccArrFormat{
		FromPrv:  fromPrvArr,
		FromAddr: fromAddrArr,
		ToPrv:    toPrvArr,
		ToAddr:   toAddrArr,
	}
	return accArrFormat
}
