package commonApi

import (
	"bytes"
	"encoding/json"
	"fmt"
	"github.com/astaxie/beego"
	"github.com/mitchellh/mapstructure"
	hashrateCommon "hashrateNode/common"
	"hashrateNode/log"
	"hashrateNode/models"
	"hashrateNode/utils"
	"io"
	"io/ioutil"
	"mime/multipart"
	"net/http"
	"strconv"
)

var npsApi string

var lastRequestAuthKeyTime int64
var authKey string
var timeStamp int64

func init() {
	npsApi = fmt.Sprintf("http://%s%s%s", beego.AppConfig.String("nps::api_host"), ":", beego.AppConfig.String("nps::api_port"))
}

func GetClientList() []*models.NpsClientInfo {
	getClientListApi := npsApi + hashrateCommon.NpsGetClientList
	formData := make(map[string]interface{})
	formData["start"] = "0"
	formData["limit"] = "1000"
	formData["order"] = "asc"
	request, res, err := sendNpsPostRequest(getClientListApi, formData)
	if err != nil || !request {
		log.Error("Nps send get client list req err:", err)
		return nil
	}
	npsClientArr := make([]*models.NpsClientInfo, 0)
	marshal, err := json.Marshal(res["rows"])
	if err != nil {
		log.Error("GetClientList json marshal error:", err)
		return nil
	}
	err = json.Unmarshal(marshal, &npsClientArr)
	if err != nil {
		log.Error("GetClientList json Unmarshal error:", err)
		return nil
	}
	return npsClientArr
}

func GetTunnel(clientId string) []*models.TcpTunnelInfo {
	getTunnelsApi := npsApi + hashrateCommon.NpsGetTunnel
	formData := make(map[string]interface{})
	formData["client_id"] = clientId
	formData["type"] = "tcp"
	formData["offset"] = "0"
	formData["limit"] = "1000"
	formData["order"] = "asc"
	request, resMap, err := sendNpsPostRequest(getTunnelsApi, formData)
	if !request || err != nil {
		log.Errorf("Nps AddTunnel error, error:%s", err)
		return nil
	}
	res := make([]*models.TcpTunnelInfo, 0)
	tunnelList, err := json.Marshal(resMap["rows"])
	if err != nil {
		log.Error("GetTunnel json marshal error:", err)
		return nil
	}
	err = json.Unmarshal(tunnelList, &res)
	if err != nil {
		log.Error("GetTunnel json unmarshal error:", err)
		return nil
	}
	return res
}

func AddTunnel(tunnelRequest *models.AddTunnelRequest) *models.NpsOperateResp {
	addTunnelApi := npsApi + hashrateCommon.NpsAddTunnel
	mapReq := make(map[string]interface{})
	marshal, err := json.Marshal(tunnelRequest)
	if err != nil {
		return nil
	}
	err = json.Unmarshal(marshal, &mapReq)
	if err != nil {
		return nil
	}
	request, resMap, err := sendNpsPostRequest(addTunnelApi, mapReq)
	if !request || err != nil {
		log.Errorf("Nps AddTunnel error, isSuccess:%s error:%s", request, err)
		return nil
	}
	res := &models.NpsOperateResp{}
	err = mapstructure.Decode(resMap, res)
	if err != nil {
		log.Errorf("Nps AddTunnel mapStructure decode error:", err)
		return nil
	}
	return res
}

func getAuthKey() (string, int64) {
	nowTimestampUri := npsApi + hashrateCommon.NpsGetTimeAPI
	authKeyUri := npsApi + hashrateCommon.NpsGetAuthKeyAPI
	isSuccess, res, err := sendNpsPostRequest(nowTimestampUri, nil)
	if err != nil || !isSuccess {
		return "", 0
	}
	timeStruct := &models.UnixTime{}
	marshalTimeStruct, err := json.Marshal(res)
	if err != nil {
		return "", 0
	}
	err = json.Unmarshal(marshalTimeStruct, timeStruct)
	if err != nil {
		return "", 0
	}
	isSuccess, res, err = sendNpsPostRequest(authKeyUri, nil)
	if err != nil || !isSuccess {
		return "", 0
	}
	authKeyStruct := &models.AuthKey{}
	marshalAuthKeyStruct, err := json.Marshal(res)
	if err != nil {
		return "", 0
	}
	err = json.Unmarshal(marshalAuthKeyStruct, authKeyStruct)
	if err != nil {
		return "", 0
	}
	return utils.GetAuthKey(authKeyStruct.CryptAuthKey, timeStruct.Timestamp), timeStruct.Timestamp
}

func sendNpsPostRequest(uri string, formData map[string]interface{}) (bool, map[string]interface{}, error) {
	payload := &bytes.Buffer{}
	writer := multipart.NewWriter(payload)
	if formData != nil {
		for key, value := range formData {
			if strValue, ok := value.(string); ok {
				_ = writer.WriteField(key, strValue)
			}
		}
		authKey, timeStamp = getAuthKey()
		if authKey == "" || timeStamp == 0 {
			return false, nil, nil
		}
		_ = writer.WriteField("auth_key", authKey)
		_ = writer.WriteField("timestamp", strconv.FormatInt(timeStamp, 10))
		lastRequestAuthKeyTime = timeStamp
		log.Info("Get auth key timestamp:", lastRequestAuthKeyTime)
	}
	err := writer.Close()
	if err != nil {
		fmt.Println("Error closing writer:", err)
		return false, nil, err
	}
	request, err := http.NewRequest("POST", uri, payload)
	request.Header.Set("Content-Type", writer.FormDataContentType())
	if err != nil {
		log.Error("Nps api new request error:", err)
		return false, nil, nil
	}
	resp, err := client.Do(request)
	if err != nil {
		log.Errorf("Nps api send post request uri:%s , error:%s", uri, err)
		return false, nil, nil
	}
	defer func(Body io.ReadCloser) {
		err := Body.Close()
		if err != nil {
			log.Error("Nps api close body error:", err)
			return
		}
	}(resp.Body)
	if resp.StatusCode != hashrateCommon.SuccessCode {
		log.Error("Nps api Send post request failed:", resp.Status)
		return false, nil, nil
	}
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return false, nil, err
	}
	var res map[string]interface{}
	err = json.Unmarshal(body, &res)
	if err != nil {
		log.Error("Nps api unmarshal resp error:", err)
		return false, nil, err
	}
	return true, res, nil
}
