package asyncLog

import (
	"errors"
	"fmt"
	"github.com/astaxie/beego"
	"github.com/astaxie/beego/logs"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
	"hashrateNode/cache"
	hashrateCommon "hashrateNode/common"
	"hashrateNode/commonApi"
	vm "hashrateNode/contract/VmContract"
	vmCreate "hashrateNode/contract/VmCreateContract"
	"hashrateNode/log"
	"hashrateNode/models"
	"hashrateNode/utils"
	"math/big"
	"strconv"
	"strings"
	"time"
)

var tranOpts *bind.TransactOpts
var hostName, fromAddr, vncDefaultPort, sshDefaultPort string
var nonce uint64
var err error

func init() {
	tranOpts, err = utils.GetTranOpts()
	if err != nil {
		log.Error("GetTranOpts error:", err)
		return
	}
	hostName = beego.AppConfig.String("pve::host_name")
	fromAddr = beego.AppConfig.String("sendTranAddress")
	vncDefaultPort = beego.AppConfig.String("vncDefaultPort")
	sshDefaultPort = beego.AppConfig.String("vncDefaultPort")
}

func VmCreateContractHandler(vLog types.Log) error {
	logs.Info("handle vm create contract logs.")
	stakeAbiInfo, _ := abi.JSON(strings.NewReader(vmCreate.VmCreateABI))
	{
		method := vLog.Topics[0]
		switch method.String() {
		case hashrateCommon.VmCreateEvent:
			{
				rp, err := stakeAbiInfo.Unpack("AddCreateVMInfoEvent", vLog.Data)
				sockets := new(big.Int).SetUint64(rp[4].(uint64))
				cores := new(big.Int).SetUint64(rp[5].(uint64))
				gpuNum := new(big.Int).SetUint64(rp[7].(uint64))
				memory := new(big.Int).SetUint64(rp[8].(uint64))
				vmCreateEvent := &models.VmConfig{
					TaskId:   rp[0].(string),
					Owner:    rp[1].(common.Address),
					CallerIp: rp[2].(string),
					Cpu:      rp[3].(string),
					Sockets:  sockets,
					Cores:    cores,
					Gpu:      rp[6].(string),
					GpuNum:   gpuNum,
					Memory:   memory,
					OsType:   rp[9].(string),
					Disk:     bigHundred,
					Name:     rp[10].(string),
				}
				//vmCreateEvent := &models.VmConfig{}
				//err = stakeAbiInfo.UnpackIntoInterface(vmCreateEvent, "AddCreateVMInfoEvent", vLog.Data)
				if err != nil {
					err := cache.Redis.LpushByte(hashrateCommon.EventList, utils.Marshal(vmCreateEvent))
					if err != nil {
						log.Error("cache redis lPush Byte error:", err.Error())
						return err
					}
				}
				if !IsCreateVmCondition(vmCreateEvent.CallerIp) {
					return nil
				}
				isSnatch, err := snatchTask(vmCreateEvent)
				if err != nil || !isSnatch {
					return err
				}
			}
		}
	}
	return nil
}

// IsCreateVmCondition 本机是否符合创建Vm的要求
func IsCreateVmCondition(callerIp string) bool {
	localIp := beego.AppConfig.String("local_ip")
	distance := utils.CalculateDistance(utils.GetIpAddr(callerIp), utils.GetIpAddr(localIp))
	if distance <= 1000 {
		return true
	}
	return true
}

// snatchTask 进行抢占任务
func snatchTask(vmCfg *models.VmConfig) (bool, error) {
	rpcUrl := beego.AppConfig.String("chain_rpc_url")
	client, err := ethclient.Dial(rpcUrl)
	if err != nil {
		log.Error("Eth client.Dial error:", err)
		return false, err
	}
	defer client.Close()
	nonce, err = utils.EthGetTransactionCount(client, fromAddr)
	if err != nil {
		return false, err
	}
	vmCreateInstance, err := vmCreate.NewVmCreate(common.HexToAddress(hashrateCommon.VmCreateContract), client)
	if err != nil {
		log.Error("New vmCreate error:", err)
		return false, err
	}
	// 查询 创建vm的操作系统对应的vmId
	updateTranOptsNonce(0)
	// 调用合约抢占任务
	task, err := vmCreateInstance.SnatchTask(tranOpts, vmCfg.TaskId, vmCfg.Owner)
	if err != nil {
		log.Error("SnatchTask error:", err)
		return false, err
	}
	log.Info("SnatchTask success:", task.Hash())

	ticket, err := commonApi.GetCreateVmTicket()
	if err != nil {
		return false, err
	}
	vmId, newId, supportType := getCloneVmInfo(vmCfg.OsType, ticket)
	if vmId == "" || newId == "" {
		return false, nil
	}
	user, password := generateUser()
	vmInfo := &models.VM{
		VmId:        vmId,
		NewId:       newId,
		VmCfg:       vmCfg,
		SupportType: supportType,
		User:        user,
		Password:    password,
		HeaderInfo:  ticket,
	}
	updateTranOptsNonce(1)
	updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(1), bigTwenty)

	// 创建VM、并且设置配置
	_, err = commonApi.CreateVm(vmInfo)
	if err != nil {
		log.Error("Create vm error:", err.Error())
		return false, err
	}
	setRes, err := commonApi.SetVmConfig(vmInfo)
	if err != nil || setRes == nil {
		log.Error("Set vm config error:", err.Error())
		return false, err
	}

	updateTranOptsNonce(1)
	updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(2), bigThirty)

	// 开启虚拟机
	_, err = commonApi.StartVm(vmInfo)
	if err != nil {
		log.Error("Start vm error:", err.Error())
		return false, err
	}
	updateTranOptsNonce(1)
	updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(3), bigForty)

	// 轮训查询状态
	for {
		vmStatus, err := commonApi.VmStatus(vmInfo)
		if err != nil {
			return false, err
		}
		if vmStatus.Status == hashrateCommon.Running {
			updateTranOptsNonce(1)
			updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(4), bigFifty)
			break
		}
		time.Sleep(time.Second * 2)
	}
	time.Sleep(time.Second * 60)
	netWorks, err := commonApi.GetVmNetWork(vmInfo)
	if err != nil {
		log.Error("Get vm net work info error:", err.Error())
		return false, err
	}
	updateTranOptsNonce(1)
	updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(5), bigEighty)

	_, err = commonApi.SetVmLoginUser(vmInfo)
	if err != nil {
		log.Error("Set vm login user error:", err.Error())
		return false, err
	}
	updateTranOptsNonce(1)
	updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(5), bigNinety)

	updateRes, err := updateVmNetWorkInfo(client, netWorks, vmInfo)
	if err != nil {
		log.Error("update vm net work info error", err.Error())
		return updateRes, err
	}
	updateTranOptsNonce(1)
	updateVmProgress(vmCreateInstance, vmInfo.VmCfg.TaskId, big.NewInt(5), bigHundred)
	return true, nil
}

// updateVmNetWorkInfo  更新vm的网络信息
func updateVmNetWorkInfo(client *ethclient.Client, networkInfos []*models.NetworkInfo, vmInfo *models.VM) (bool, error) {
	internalIp := ""
	for _, netWork := range networkInfos {
		//if netWork.Name == hashrateCommon.Ens18 {
		for _, ipInfo := range netWork.IpAddresses {
			if ipInfo.IpAddressType == hashrateCommon.Ipv4 && ipInfo.IpAddress != hashrateCommon.LocalHost {
				internalIp = ipInfo.IpAddress
				log.Info("internalIp:", internalIp)
				break
			}
		}
		//}
	}
	if internalIp == "" {
		log.Error("In networkInfo is not found ip info")
		return false, errors.New("internalIp is not found")
	}
	vmContract, err := vm.NewVm(common.HexToAddress(hashrateCommon.VmContract), client)
	if err != nil {
		log.Error("NewVm failed", err.Error())
		return false, err
	}
	// 调用NPS 映射外网ip
	vmNetWorkInfo := npsMappingNetwork(internalIp, vmInfo.SupportType)
	if vmNetWorkInfo == nil {
		log.Error("npsMappingNetwork is failed")
		return false, errors.New("vmNetWorkInfo is not found")
	}
	vmId := new(big.Int)
	vmId.SetString(vmInfo.NewId, 10)
	updateTranOptsNonce(1)
	addVirtualMachine, err := vmContract.AddVirtualMachine(tranOpts, vmInfo.VmCfg.Owner, vmId, "", "", hostName, vmInfo.VmCfg.Name, vmInfo.VmCfg.OsType)
	if err != nil {
		log.Error("addVirtualMachine failed", err.Error())
		return false, err
	}
	log.Info("Add virtual machine success:", addVirtualMachine.Hash())

	updateTranOptsNonce(1)
	updateVmNetworkInfo, err := vmContract.UpdateVmNetworkInfo(tranOpts, vmId, vmNetWorkInfo.InternalIp, vmNetWorkInfo.ExternalIp, vmNetWorkInfo.ExSshPort, vmNetWorkInfo.ExVncPort, strconv.FormatInt(vmNetWorkInfo.InSshPort, 10), strconv.FormatInt(vmNetWorkInfo.InVncPort, 10))
	if err != nil {
		log.Error("Update virtual machine", err.Error())
		return false, err
	}
	log.Info("Update virtual machine  networkInfo success:", updateVmNetworkInfo.Hash())

	updateTranOptsNonce(1)
	updateVMInfoEvent, err := vmContract.UpdateVmInfo(tranOpts, vmId, vmInfo.VmCfg.Sockets.Uint64(), vmInfo.VmCfg.Memory.Uint64(), vmInfo.VmCfg.GpuNum.Uint64(), vmInfo.VmCfg.Disk)
	if err != nil {
		log.Error("Update vm info", err.Error())
		return false, err
	}
	log.Info("Update virtual machine info success:", updateVMInfoEvent.Hash())

	updateTranOptsNonce(1)
	userTran, err := vmContract.AddVmUser(tranOpts, vmId, vmInfo.User, vmInfo.Password)
	if err != nil {
		log.Error("Add vm user error:", err)
		return false, err
	}
	log.Info("Update virtual machine info success:", userTran.Hash())
	return true, nil
}

func npsMappingNetwork(internalIp string, supportType int) *models.VmNetWork {
	// 获取本机的外网Ip
	externalIp := getExternalIp()
	// 获取可用的客户端
	clientList := commonApi.GetClientList()
	if len(clientList) == 0 {
		addClient()
		return nil
	}
	tunnelModel := &models.AddTunnelRequest{
		Type: "tcp",
	}
	for _, clientInfo := range clientList {
		if clientInfo.Status && clientInfo.IsConnect {
			tunnelModel.ClientId = strconv.FormatInt(clientInfo.Id, 10)
			break
		}
	}
	vmNetWorkInfo := &models.VmNetWork{
		InternalIp:  internalIp,
		InSshPort:   hashrateCommon.DefaultSshPort,
		InVncPort:   hashrateCommon.DefaultVncPort,
		ExternalIp:  externalIp,
		SupportType: int64(supportType),
	}
	unUsePort := getUnUsePort(tunnelModel.ClientId)
	tunnelModel.Port = strconv.FormatInt(unUsePort, 10)
	if supportType == hashrateCommon.VmSupportConnectVnc {
		tunnelModel.Target = fmt.Sprintf("%s:%s", internalIp, vncDefaultPort)
		vmNetWorkInfo.ExVncPort = tunnelModel.Port
	} else if supportType == hashrateCommon.VmSupportConnectSsh {
		tunnelModel.Target = fmt.Sprintf("%s:%s", internalIp, sshDefaultPort)
		vmNetWorkInfo.ExSshPort = tunnelModel.Port
	}

	if supportType == hashrateCommon.VmSupportConnectAll {
		// 处理SSH端口
		tunnelModel.Target = fmt.Sprintf("%s:%s", internalIp, sshDefaultPort)
		addRes := commonApi.AddTunnel(tunnelModel)
		if addRes.Status == hashrateCommon.NpsReqFailedStatus {
			log.Error("Add ssh tunnel failed error msg:", addRes.Msg)
			return nil
		}
		vmNetWorkInfo.ExSshPort = tunnelModel.Port

		// 处理VNC端口
		tunnelModel.Port = strconv.FormatInt(unUsePort+1, 10)
		tunnelModel.Target = fmt.Sprintf("%s:%s", internalIp, vncDefaultPort)
		addRes = commonApi.AddTunnel(tunnelModel)
		if addRes.Status == hashrateCommon.NpsReqFailedStatus {
			log.Error("Add vnc tunnel failed error msg:", addRes.Msg)
			return nil
		}
		vmNetWorkInfo.ExVncPort = tunnelModel.Port
	} else {
		addRes := commonApi.AddTunnel(tunnelModel)
		if addRes.Status == hashrateCommon.NpsReqFailedStatus {
			log.Error("Add tunnel failed error msg:", addRes.Msg)
			return nil
		}
	}
	return vmNetWorkInfo
}

func addClient() {

}

func getUnUsePort(clientId string) int64 {
	tunnelList := commonApi.GetTunnel(clientId)
	var maxPort int64
	for _, tunnel := range tunnelList {
		if tunnel.Port > maxPort {
			maxPort = tunnel.Port
		}
	}
	return maxPort + 1
}

func getExternalIp() string {
	externalIp := beego.AppConfig.String("nps::api_host")
	if externalIp == "" {
		// todo 如果本机没有host 外网ip、则需要通过查找网关进行分配外网ip
		return externalIp
	}
	return externalIp
}

// 生成用户
func generateUser() (string, string) {
	return "wuban", "2023!@#WSX#@!"
}

// getCloneVmInfo 获取能够克隆的vmId
func getCloneVmInfo(osType string, headerInfo *models.HeaderInfo) (string, string, int) {
	qemuList, err := commonApi.GetQemuList(headerInfo)
	if err != nil {
		log.Error("Get qemu list error:", err.Error())
		return "", "", 0
	}
	var maxVmId int64
	supportType := 0
	var cloneId string
	for _, qemu := range qemuList {
		if qemu.VmId > maxVmId {
			maxVmId = qemu.VmId
		}
		if qemu.Template == 1 {
			templateInfo := strings.Split(qemu.Name, hashrateCommon.SplitSupportType)
			osNameTem := templateInfo[0]
			versionTem := templateInfo[1]
			supportTypeTem := templateInfo[2]
			supportType, err = strconv.Atoi(supportTypeTem)
			if err != nil {
				log.Error("strconv os support type type error:", err.Error())
				return "", "", 0
			}
			osInfo := strings.Split(osType, hashrateCommon.SplitSupportType)
			osNameReq := osInfo[0]
			versionReq := osInfo[1]
			if osNameReq == osNameTem && versionTem == versionReq {
				cloneId = strconv.FormatInt(qemu.VmId, 10)
			}
		}
	}
	maxVmId++
	return cloneId, strconv.FormatInt(maxVmId, 10), supportType
}

// updateVmProgress  更新vm的创建进度
func updateVmProgress(contractInstance *vmCreate.VmCreate, taskId string, stage *big.Int, progress *big.Int) bool {
	task, err := contractInstance.UpdateVmCreateProgress(tranOpts, taskId, stage, progress)
	if err != nil {
		log.Error("UpdateVmCreateProgress error:", err)
		return false
	}
	log.Info("Update vm create progress success:", task.Hash())
	return true
}

// updateTranOptsNonce 	更新Nonce
func updateTranOptsNonce(incrementNum uint64) {
	nonce += incrementNum
	nonceUint64 := new(big.Int).SetUint64(nonce)
	tranOpts.Nonce = nonceUint64
}
