package controllers

import (
	"bufio"
	"context"
	"encoding/binary"
	"encoding/json"
	"github.com/astaxie/beego"
	"github.com/astaxie/beego/orm"
	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	log "github.com/sirupsen/logrus"
	"github.com/wuban/nft-event/models"
	"github.com/wuban/nft-event/utils"
	"io/ioutil"
	"math/big"
	"net/http"
	"os"
	"strings"
)

type LotteryController struct {
	BaseController
}

func (c *LotteryController) GetTokenIdList() {
	poolAddr := c.GetString("poolAddr")
	contractAddr := c.GetString("lotteryContract")
	if !utils.IsNilString(contractAddr, poolAddr) || !utils.IsAddress(poolAddr, contractAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
		return
	}
	o := orm.NewOrm()
	var lotteries []*models.Lottery
	_, err := o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("pool_addr", poolAddr).Distinct().OrderBy("period").All(&lotteries, "token_id", "period")

	if err != nil {
		c.ResponseInfo(500, models.PERIOD_ERR, err.Error())
		return
	}
	res := &models.LotteryPeriodListRes{
		PoolAddr: poolAddr,
	}
	tokenIdList := make([]string, 0)
	for i := 0; i < len(lotteries); i++ {
		tokenIdList = append(tokenIdList, lotteries[i].TokenId)
	}
	res.TokenIdList = tokenIdList
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetUserList() {
	poolAddr := c.GetString("poolAddr")
	contractAddr := c.GetString("lotteryContract")
	if !utils.IsNilString(contractAddr, poolAddr) || !utils.IsAddress(poolAddr, contractAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
		return
	}
	period, err := c.GetInt64("period")
	if err != nil {
		c.ResponseInfo(500, models.PARAM_ERR, err.Error())
		return
	}
	o := orm.NewOrm()
	var lotteries []*models.Lottery
	_, err = o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("pool_addr", poolAddr).Filter("period", period).OrderBy("-sort_by").All(&lotteries, "user", "value", "hash")
	if err != nil {
		c.ResponseInfo(500, models.USER_ERR, err.Error())
		return
	}
	res := make([]*models.LotteryUserList, 0)
	for i := 0; i < len(lotteries); i++ {
		lottery := lotteries[i]
		lotteryUserList := &models.LotteryUserList{
			UserAddr: lottery.User,
			Value:    lottery.Value,
			Hash:     lottery.Hash,
		}
		res = append(res, lotteryUserList)
	}
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetHashSortByCondition() {
	poolAddr := c.GetString("poolAddr")
	period := c.GetString("period")
	contractAddr := c.GetString("lotteryContract")
	if !utils.IsNilString(period, contractAddr, poolAddr) || !utils.IsAddress(poolAddr, contractAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	o := orm.NewOrm()
	lotteries := make([]models.Lottery, 0)
	_, err := o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("poolAddr", poolAddr).Filter("period", period).OrderBy("-sort_by").All(&lotteries)
	if err != nil {
		c.ResponseInfo(500, models.SORTBY_ERR, err.Error())
		return
	}
	if len(lotteries) == 0 {
		c.ResponseInfo(200, models.DATA_NIL, nil)
		return
	}
	lotteryHashLastSix := &models.LotteryHashLastSix{
		Hash:   lotteries[0].Hash,
		SortBy: lotteries[0].SortBy,
	}
	c.ResponseInfo(200, models.SUCCESS, lotteryHashLastSix)
}

func (c *LotteryController) ForwardReq() {
	nftAddr := c.GetString("nftAddr")
	id := c.GetString("id")
	regex := utils.GetHttpRegex()
	if !utils.IsNilString(id, nftAddr) || regex == nil || regex.MatchString(id) || regex.MatchString(nftAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	rpc := beego.AppConfig.String("rpcUrl")
	ethClient, err := ethclient.Dial(rpc)
	if err != nil {
		log.Error(models.DIAL_RPC_ERR, err.Error())
		c.ResponseInfo(500, models.DIAL_RPC_ERR, err.Error())
		return
	}
	if !utils.IsContractAddr(nftAddr, ethClient) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	defer ethClient.Close()
	erc721, err := utils.NewERC721(utils.HexToAddr(nftAddr), ethClient)
	if err != nil {
		c.ResponseInfo(500, models.BUILD_ERR, nil)
		return
	}
	idBigInt := big.NewInt(0)
	_, setRes := idBigInt.SetString(id, 10)
	if !setRes {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	uri, err := erc721.TokenURI(&bind.CallOpts{}, idBigInt)
	if err != nil || uri == "" {
		c.ResponseInfo(500, models.FAILED, nil)
		return
	}
	if strings.HasPrefix(uri, "ipfs://") {
		uri = beego.AppConfig.String("beforeEndReqUrl") + uri[7:]
	}
	if utils.TokenUriMap[nftAddr] != "" {
		uri = utils.TokenUriMap[nftAddr] + id
	}
	client := &http.Client{}
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
		return
	}
	resp, err := client.Do(req)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
		return
	}
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
		return
	}
	res := map[string]interface{}{}
	json.Unmarshal(body, &res)
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetPoolInfo() {
	maxHistory := c.GetString("maxHistory")
	contractStr := c.GetString("lotteryContract")
	if !utils.IsNilString(contractStr) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	rpc := beego.AppConfig.String("rpcUrl")
	client, err := ethclient.Dial(rpc)
	if err != nil {
		log.Error(models.DIAL_RPC_ERR, err.Error())
		c.ResponseInfo(500, models.DIAL_RPC_ERR, err.Error())
		return
	}
	if !utils.IsContractAddr(contractStr, client) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	defer client.Close()
	abiBytes := &os.File{}
	if maxHistory != "" {
		abiBytes, err = os.Open("./sync/Lottery_Duration.json")
	} else {
		abiBytes, err = os.Open("./sync/Lottery_Func.json")
	}
	if err != nil {
		log.Error(models.READ_ABI_ERR, err)
		c.ResponseInfo(500, models.READ_ABI_ERR, err.Error())
		return
	}
	// 解析ABI文件
	contractAbi, err := abi.JSON(bufio.NewReader(abiBytes))
	if err != nil {
		log.Error(models.PARSE_ABI_ERR, err)
		c.ResponseInfo(500, models.PARSE_ABI_ERR, err.Error())
		return
	}
	data := make([]byte, 0)
	if maxHistory != "" {
		maxHistoryParam := big.NewInt(0)
		_, setRes := maxHistoryParam.SetString(maxHistory, 10)
		if !setRes {
			c.ResponseInfo(500, models.PARAM_ERR, nil)
			return
		}
		data, err = contractAbi.Pack("getPoolInfo", maxHistoryParam)
		if err != nil {
			log.Error(models.PACK_ABI_ERR, err)
			c.ResponseInfo(500, models.PACK_ABI_ERR, err.Error())
			return
		}
	} else {
		data, err = contractAbi.Pack("getPoolInfo")
	}
	if err != nil {
		log.Error(models.PACK_ABI_ERR, err)
		c.ResponseInfo(500, models.PACK_ABI_ERR, err.Error())
		return
	}
	contractAddr := common.HexToAddress(contractStr)
	// 执行合约调用
	log.Info("data:", data)
	result, err := client.CallContract(context.TODO(), ethereum.CallMsg{
		To:   &contractAddr,
		Data: data,
	}, nil)
	if err != nil {
		log.Error(models.CALL_CONTRACT_ERR, err)
		c.ResponseInfo(500, models.CALL_CONTRACT_ERR, err.Error())
		return
	}
	res := &models.PoolInfoRes{}
	// 解析返回值
	err = contractAbi.UnpackIntoInterface(res, "getPoolInfo", result)
	if err != nil {
		log.Error(models.UNPACK_ABI_RETURN_PARM_ERR, err)
		c.ResponseInfo(500, models.UNPACK_ABI_RETURN_PARM_ERR, err.Error())
		return
	}
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetBeforePeriodWinnerAndSign() {
	poolAddr := c.GetString("poolAddr")
	contractAddr := c.GetString("lotteryContract")
	period, err := c.GetInt64("period")
	beforeNum, err := c.GetInt64("beforeNum")
	rewardNum, err := c.GetInt("rewardNum", 1)
	if err != nil || !utils.IsNilString(contractAddr, poolAddr) || period <= 0 || rewardNum <= 0 {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
		return
	}
	var startNum int64 = 0
	if period-beforeNum > 0 {
		startNum = period - beforeNum
	}
	signPrv, err := crypto.HexToECDSA(beego.AppConfig.String("signAccPrv"))
	if err != nil {
		c.ResponseInfo(200, models.FAILED, err.Error())
		return
	}
	o := orm.NewOrm()
	periodArr := make([][]int64, 0)
	winnerArr := make([][]string, 0)
	txHashArr := make([][]string, 0)
	signArr := make([][]string, 0)
	highPeriodLottery := &models.Lottery{}
	o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("pool_addr", poolAddr).OrderBy("-period").One(highPeriodLottery, "period")
	if highPeriodLottery.Period < period {
		period = highPeriodLottery.Period
	}
	if period == 0 {
		period += 1
	}
	for i := startNum; i < period; i++ {
		lotteries := make([]*models.Lottery, 0)
		o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("pool_addr", poolAddr).Filter("period", i).OrderBy("-sort_by").Limit(rewardNum).All(&lotteries)
		if len(lotteries) == 0 {
			continue
		}
		periodTemp := make([]int64, 0)
		winnerTemp := make([]string, 0)
		txHashTemp := make([]string, 0)
		signTemp := make([]string, 0)
		for j := 0; j < len(lotteries); j++ {
			lottery := lotteries[j]
			periodBytes := make([]byte, 8)
			binary.BigEndian.PutUint64(periodBytes, uint64(lottery.Period))
			winnerAddr := utils.HexToAddr(lottery.User)
			hashData := []byte(lottery.Hash)
			hash := crypto.Keccak256Hash(common.LeftPadBytes(periodBytes, 32), winnerAddr.Bytes(), hashData)
			signature, err := crypto.Sign(hash.Bytes(), signPrv)
			if err != nil {
				c.ResponseInfo(500, models.FAILED, nil)
				return
			}
			signature[64] += 27
			periodTemp = append(periodTemp, lottery.Period)
			winnerTemp = append(winnerTemp, lottery.User)
			txHashTemp = append(txHashTemp, lottery.Hash)
			log.Info(signature)
			signTemp = append(signTemp, "0x"+common.Bytes2Hex(signature))
		}
		periodArr = append(periodArr, periodTemp)
		winnerArr = append(winnerArr, winnerTemp)
		txHashArr = append(txHashArr, txHashTemp)
		signArr = append(signArr, signTemp)
	}
	winnerRes := &models.WinnerListRes{
		PoolAddr:  poolAddr,
		PeriodArr: periodArr,
		WinnerArr: winnerArr,
		TxHashArr: txHashArr,
		SignArr:   signArr,
	}
	c.ResponseInfo(200, models.SUCCESS, winnerRes)
}
