package sync

import (
	"context"
	"fmt"
	"github.com/astaxie/beego"
	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient"
	log "github.com/sirupsen/logrus"
	"github.com/wuban/nft-event/cache"
	"math/big"
	"time"
)

const (
	LastSyncBlockKey = "lastSyncBlock"
)

var (
	pullTask   *PullEvent
	bigOne     = big.NewInt(1)
	bigTen     = big.NewInt(10)
	bigHundred = big.NewInt(100)
	bigEight   = big.NewInt(700)
)

type logHandler func(log types.Log) error

type PullEvent struct {
	ctx             context.Context
	client          *ethclient.Client
	lastBlock       *big.Int
	contractList    []common.Address
	contractHandler map[common.Address]logHandler
}

func SyncLogs() {
	pullTask.GetLogs()
}

func init() {
	startBlock := beego.AppConfig.String("deployedBlock")
	cache.Redis.Set(LastSyncBlockKey, startBlock)
	var err error
	rpc := beego.AppConfig.String("rpcUrl")
	deployBlock := beego.AppConfig.String("deployedBlock")
	fmt.Println("get rpc ", rpc)
	pullTask = &PullEvent{contractHandler: make(map[common.Address]logHandler)}
	client, err := ethclient.Dial(rpc)
	if err != nil {
		panic(fmt.Sprintf("ethclient dial failed, err:", err))
	} else {
		pullTask.client = client
		lastBlock := cache.Redis.Get(LastSyncBlockKey)
		if len(lastBlock) == 0 {
			lastBlock = deployBlock
		}
		{
			blockNumber, _ := new(big.Int).SetString(lastBlock, 10)
			pullTask.lastBlock = blockNumber
		}
	}
	pullTask.ctx = context.Background()
	pullTask.contractList = make([]common.Address, 0)
	{
		lotteryAddr := beego.AppConfig.String("lotteryContract")
		if lotteryAddr == "" {
			return
		}
		addr := common.HexToAddress(lotteryAddr)
		pullTask.contractList = append(pullTask.contractList, addr)
		pullTask.contractHandler[addr] = LotteryContractHandler
	}
}

func (p *PullEvent) GetLogs() {
	backBlock, err := beego.AppConfig.Int64("goBackBlockNum")
	if err != nil {
		log.Error("Get backBlock error:", err.Error())
		return
	}
	query := ethereum.FilterQuery{}
	query.FromBlock = p.lastBlock
	query.ToBlock = new(big.Int).Add(p.lastBlock, big.NewInt(1))
	query.Addresses = p.contractList
	for {
		query.FromBlock = p.lastBlock
		height, err := p.client.BlockNumber(p.ctx)
		log.Info("Current node height:", height)
		if height < p.lastBlock.Uint64() {
			time.Sleep(time.Second)
			continue
		} else if (height - 700) >= p.lastBlock.Uint64() {
			query.ToBlock = new(big.Int).Add(p.lastBlock, bigEight)
		} else if (height - 100) >= p.lastBlock.Uint64() {
			query.ToBlock = new(big.Int).Add(p.lastBlock, bigHundred)
		} else if (height - 10) >= p.lastBlock.Uint64() {
			query.ToBlock = new(big.Int).Add(p.lastBlock, bigTen)
		} else {
			query.FromBlock = new(big.Int).Sub(p.lastBlock, big.NewInt(backBlock))
			query.ToBlock = p.lastBlock
		}
		log.Infof("start filter bloct at:%s, end filter block at:%s", query.FromBlock.Text(10), query.ToBlock.Text(10))
		allLogs, err := p.client.FilterLogs(p.ctx, query)
		if err != nil {
			log.Error("filter logs failed", err)
			continue
		}
		log.Info("Filter log length:", len(allLogs))
		if len(allLogs) > 0 {
			for _, vlog := range allLogs {
				log.Info("Logs to be processed:", vlog)
				handle, exist := p.contractHandler[vlog.Address]
				if exist {
					handle(vlog)
				}
			}
		}
		p.lastBlock = new(big.Int).Add(query.ToBlock, bigOne)
		cache.Redis.Set(LastSyncBlockKey, p.lastBlock.Text(10))
	}
}
