package controllers

import (
	"encoding/json"
	"github.com/astaxie/beego"
	"github.com/astaxie/beego/orm"
	"github.com/wuban/nft-event/models"
	"github.com/wuban/nft-event/utils"
	"io/ioutil"
	"net/http"
	"regexp"
)

type LotteryController struct {
	BaseController
}

func (c *LotteryController) GetPeriodList() {
	poolAddr := c.GetString("poolAddr")
	if poolAddr == "" || !utils.IsAddress(poolAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
	}
	o := orm.NewOrm()
	var lotteries []*models.Lottery
	_, err := o.QueryTable("lottery").Filter("pool_addr", poolAddr).All(&lotteries, "period")
	if err != nil {
		c.ResponseInfo(500, models.PERIOD_ERR, err.Error())
	}
	res := &models.LotteryPeriodListRes{
		PoolAddr: poolAddr,
	}
	periodList := make([]int64, 0)
	for i := 0; i < len(lotteries); i++ {
		periodList = append(periodList, lotteries[i].Period)
	}
	res.PeriodList = periodList
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetUserList() {
	poolAddr := c.GetString("poolAddr")
	if poolAddr == "" || !utils.IsAddress(poolAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
	}
	period, err := c.GetInt64("period")
	if err != nil {
		c.ResponseInfo(500, models.PARAM_ERR, err.Error())
	}
	o := orm.NewOrm()
	var lotteries []*models.Lottery
	_, err = o.QueryTable("lottery").Filter("pool_addr", poolAddr).Filter("period", period).OrderBy("-sort_by").All(&lotteries, "user", "value", "hash")
	if err != nil {
		c.ResponseInfo(500, models.USER_ERR, err.Error())
	}
	res := make([]*models.LotteryUserList, 0)
	for i := 0; i < len(lotteries); i++ {
		lottery := lotteries[i]
		lotteryUserList := &models.LotteryUserList{
			UserAddr: lottery.User,
			Value:    lottery.Value,
			Hash:     lottery.Hash,
		}
		res = append(res, lotteryUserList)
	}
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetHashSortByCondition() {
	poolAddr := c.GetString("poolAddr")
	if poolAddr == "" || !utils.IsAddress(poolAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
	}
	userAddr := c.GetString("user")
	if userAddr == "" || !utils.IsAddress(poolAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
	}
	period, err := c.GetInt64("period")
	if err != nil {
		c.ResponseInfo(500, models.PARAM_ERR, err.Error())
	}
	o := orm.NewOrm()
	lottery := &models.Lottery{}
	err = o.QueryTable("lottery").Filter("pool_addr", poolAddr).Filter("period", period).One(lottery)
	if err != nil {
		c.ResponseInfo(500, models.SORTBY_ERR, err.Error())
	}
	if lottery == nil {
		c.ResponseInfo(200, models.DATA_NIL, nil)
	}
	lotteryHashLastSix := &models.LotteryHashLastSix{
		Hash:   lottery.Hash,
		SortBy: lottery.SortBy,
	}
	c.ResponseInfo(200, models.SUCCESS, lotteryHashLastSix)
}

func (c *LotteryController) ForwardReq() {
	id := c.GetString("id")
	urlPattern := beego.AppConfig.String("httpUrlRegex")
	// 编译正则表达式
	regex, err := regexp.Compile(urlPattern)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	if id == "" {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
	}
	if regex.MatchString(id) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
	}
	url := beego.AppConfig.String("beforeEndReqUrl") + id
	client := &http.Client{}
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	resp, err := client.Do(req)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	res := map[string]interface{}{}
	json.Unmarshal(body, &res)
	c.ResponseInfo(200, models.SUCCESS, res)
}
