package controllers

import (
	"encoding/json"
	"github.com/astaxie/beego"
	"github.com/astaxie/beego/orm"
	"github.com/wuban/nft-event/models"
	"github.com/wuban/nft-event/utils"
	"io/ioutil"
	"net/http"
	"regexp"
)

type LotteryController struct {
	BaseController
}

func (c *LotteryController) GetTokenIdList() {
	poolAddr := c.GetString("poolAddr")
	if poolAddr == "" || !utils.IsAddress(poolAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
	}
	contractAddr := beego.AppConfig.String("lotteryContract")
	o := orm.NewOrm()
	var lotteries []*models.Lottery
	_, err := o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("pool_addr", poolAddr).Distinct().OrderBy("period").All(&lotteries, "token_id", "period")

	if err != nil {
		c.ResponseInfo(500, models.PERIOD_ERR, err.Error())
	}
	res := &models.LotteryPeriodListRes{
		PoolAddr: poolAddr,
	}
	tokenIdList := make([]string, 0)
	for i := 0; i < len(lotteries); i++ {
		tokenIdList = append(tokenIdList, lotteries[i].TokenId)
	}
	res.TokenIdList = tokenIdList
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetUserList() {
	poolAddr := c.GetString("poolAddr")
	if poolAddr == "" || !utils.IsAddress(poolAddr) {
		c.ResponseInfo(500, models.PARAM_ERR, "")
	}
	period, err := c.GetInt64("period")
	if err != nil {
		c.ResponseInfo(500, models.PARAM_ERR, err.Error())
	}
	contractAddr := beego.AppConfig.String("lotteryContract")
	o := orm.NewOrm()
	var lotteries []*models.Lottery
	_, err = o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("pool_addr", poolAddr).Filter("period", period).OrderBy("-sort_by").All(&lotteries, "user", "value", "hash")
	if err != nil {
		c.ResponseInfo(500, models.USER_ERR, err.Error())
	}
	res := make([]*models.LotteryUserList, 0)
	for i := 0; i < len(lotteries); i++ {
		lottery := lotteries[i]
		lotteryUserList := &models.LotteryUserList{
			UserAddr: lottery.User,
			Value:    lottery.Value,
			Hash:     lottery.Hash,
		}
		res = append(res, lotteryUserList)
	}
	c.ResponseInfo(200, models.SUCCESS, res)
}

func (c *LotteryController) GetHashSortByCondition() {
	period := c.GetString("period")
	if period == "" {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
	}
	contractAddr := beego.AppConfig.String("lotteryContract")
	o := orm.NewOrm()
	lotteries := make([]models.Lottery, 0)
	_, err := o.QueryTable("lottery").Filter("contract_addr", contractAddr).Filter("period", period).OrderBy("-sort_by").All(&lotteries)
	if err != nil {
		c.ResponseInfo(500, models.SORTBY_ERR, err.Error())
	}
	if len(lotteries) == 0 {
		c.ResponseInfo(200, models.DATA_NIL, nil)
	}
	lotteryHashLastSix := &models.LotteryHashLastSix{
		Hash:   lotteries[0].Hash,
		SortBy: lotteries[0].SortBy,
	}
	c.ResponseInfo(200, models.SUCCESS, lotteryHashLastSix)
}

func (c *LotteryController) ForwardReq() {
	id := c.GetString("id")
	urlPattern := beego.AppConfig.String("httpUrlRegex")
	// 编译正则表达式
	regex, err := regexp.Compile(urlPattern)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	if id == "" {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
	}
	if regex.MatchString(id) {
		c.ResponseInfo(500, models.PARAM_ERR, nil)
	}
	url := beego.AppConfig.String("beforeEndReqUrl") + id
	client := &http.Client{}
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	resp, err := client.Do(req)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		c.ResponseInfo(500, models.FAILED, err.Error())
	}
	res := map[string]interface{}{}
	json.Unmarshal(body, &res)
	c.ResponseInfo(200, models.SUCCESS, res)
}
