package utils

import (
	"fmt"
	"github.com/astaxie/beego"
	"github.com/astaxie/beego/context"
	"github.com/dgrijalva/jwt-go"
	"github.com/wuban/nft-event/models"
	"net/http"
	"time"
)

// 定义 JWT 密钥
var jwtKey = []byte(beego.AppConfig.String("secret"))

// Claims 定义自定义声明
type Claims struct {
	Address string `json:"address"`
	jwt.StandardClaims
}

// GenerateToken 生成 JWT
func GenerateToken(address string) (string, error) {
	// 设置过期时间
	expirationTime := time.Now().Add(15 * time.Minute)

	// 创建声明
	claims := &Claims{
		Address: address,
		StandardClaims: jwt.StandardClaims{
			ExpiresAt: expirationTime.Unix(),
			IssuedAt:  time.Now().Unix(),
		},
	}

	// 创建令牌
	token := jwt.NewWithClaims(jwt.SigningMethodHS256, claims)

	// 签名令牌
	tokenString, err := token.SignedString(jwtKey)
	if err != nil {
		return "", err
	}

	return tokenString, nil
}

// AuthFilter 鉴权过滤器
var AuthFilter = func(ctx *context.Context) {
	// 判断路由是否需要鉴权
	if ctx.Request.URL.Path == "/getToken" {
		// 不进行鉴权校验，直接通过
		return
	}
	filterRes := &models.FilterRes{
		Data: "",
		Code: http.StatusUnauthorized,
		Msg:  "Unauthorized",
	}
	authHeader := ctx.Input.Header("Authorization")
	if authHeader == "" {
		ctx.Output.SetStatus(http.StatusUnauthorized)
		ctx.Output.JSON(filterRes, true, false)
		return
	}

	// 解析 JWT
	tokenString := authHeader[len("Bearer "):]
	claims := &Claims{}

	token, err := jwt.ParseWithClaims(tokenString, claims, func(token *jwt.Token) (interface{}, error) {
		if _, ok := token.Method.(*jwt.SigningMethodHMAC); !ok {
			return nil, fmt.Errorf("UnExpected signing method")
		}
		return jwtKey, nil
	})

	if err != nil {
		if err == jwt.ErrSignatureInvalid {
			ctx.Output.SetStatus(http.StatusUnauthorized)
			ctx.Output.JSON(filterRes, true, false)
			return
		}
		filterRes.Code = http.StatusBadRequest
		filterRes.Msg = "Bad Request"
		ctx.Output.SetStatus(http.StatusBadRequest)
		ctx.Output.JSON(filterRes, true, false)
		return
	}

	if !token.Valid {
		ctx.Output.SetStatus(http.StatusUnauthorized)
		ctx.Output.JSON(filterRes, true, false)
		return
	}

	if claims.Address != beego.AppConfig.String("secret") {
		ctx.Output.SetStatus(http.StatusUnauthorized)
		filterRes.Msg = "Error token"
		ctx.Output.JSON(filterRes, true, false)
		return
	}
}
