package com.wuban.signDemo;

import com.wuban.util.CustomizeHttpService;
import org.web3j.crypto.*;
import org.web3j.protocol.Web3j;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.protocol.core.methods.response.EthGetTransactionReceipt;
import org.web3j.protocol.core.methods.response.EthSendTransaction;
import org.web3j.tx.RawTransactionManager;
import org.web3j.utils.Numeric;
import java.math.BigInteger;
import java.util.Objects;
import java.util.concurrent.ExecutionException;

/**
 * @ClassName SignAndSubmit
 * @Author DuanJinFei
 * @Date 2022/8/22 17:58
 * @Version 1.0
 */
public class SignAndSubmit {
    private static Web3j web3j;

    public SignAndSubmit(String rpcUrl) {
        web3j = Web3j.build(new CustomizeHttpService(rpcUrl));
    }
    /**
     * 提交交易
     * @param signedMessage 交易签名
     */
    public void submitSign(byte[] signedMessage) {
        try {
            String hexValue = Numeric.toHexString(signedMessage);
            EthSendTransaction ethSendTransaction = web3j.ethSendRawTransaction(hexValue).sendAsync().get();
            // 交易Hash
            String transactionHash = ethSendTransaction.getTransactionHash();
            System.out.println("transactionHash:"+transactionHash);
            if (ethSendTransaction.hasError()) {
                String message = ethSendTransaction.getError().getMessage();
                System.out.println("transaction failed,info:" + message);
            } else {
                EthGetTransactionReceipt send = web3j.ethGetTransactionReceipt(transactionHash).send();
                if (send != null) {
                    System.out.println("交易成功");
                }
            }
        } catch (Exception e) {
            if (e instanceof ExecutionException || e instanceof InterruptedException) {
                System.out.println("----------获取Nonce异常-----------");
            }
            e.printStackTrace();
        }
    }

    /**
     * 获取账户交易Nonce
     * @param privateKey 账户地址
     * @return 返回Nonce
     */
    public BigInteger getAddressNonce(String privateKey) {
        try {
            String fromAddress = Keys.toChecksumAddress(
                    Keys.getAddress(
                            ECKeyPair.create(
                                    Numeric.toBigInt(privateKey))));
            System.out.println("fromAddress:"+fromAddress);
            return web3j.ethGetTransactionCount(
                    fromAddress, DefaultBlockParameterName.LATEST).send().getTransactionCount();
        } catch (Exception e) {
            e.printStackTrace();
            return BigInteger.valueOf(-1);
        }
    }
    /**
     * 私钥交易
     * @param fromPk    发送方私钥
     * @param toAddress 接收方地址
     * @param value     数量 单位wei
     */
    public byte[] offlineSign(String fromPk, String toAddress, BigInteger value) {
        try {
            BigInteger nonce = getAddressNonce(fromPk);
            System.out.println("Nonce:"+nonce);
            BigInteger gasPrice = web3j.ethGasPrice().send().getGasPrice();
            BigInteger gasLimit = new BigInteger("900000");
            if (fromPk.startsWith("0x")){
                fromPk = fromPk.substring(2);
            }
            Credentials credentials = Credentials.create(fromPk);
            //生成RawTransaction交易对象
            RawTransaction rawTransaction = RawTransaction.createTransaction(
                    nonce,
                    gasPrice,
                    gasLimit,
                    toAddress,
                    value,
                    "");
            //使用私钥生成Credentials对象
            System.out.println("toAddress:"+toAddress);
            return TransactionEncoder.signMessage(rawTransaction,512512, credentials);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 签名且提交
     * @param fromPk    发送方私钥
     * @param toAddress 接收方地址
     * @param value     数量 单位wei
     */
    public boolean signAndSend(String fromPk, String toAddress, BigInteger value) {
        try {
            BigInteger nonce = getAddressNonce(fromPk);
            System.out.println("Nonce:"+nonce);
            BigInteger gasPrice = web3j.ethGasPrice().send().getGasPrice();
            BigInteger gasLimit = new BigInteger("900000");
            Credentials credentials = Credentials.create(fromPk);
            //生成RawTransaction交易对象
            RawTransactionManager transactionManager = new RawTransactionManager(web3j,
                    credentials,
                    512512);
            RawTransaction rawTransaction = RawTransaction.createTransaction(
                    nonce,
                    gasPrice,
                    gasLimit,
                    toAddress,
                    value,
                    "");
            //使用私钥生成Credentials对象
            System.out.println("toAddress:"+toAddress);
            EthSendTransaction ethSendTransaction = transactionManager.signAndSend(rawTransaction);
            String transactionHash = ethSendTransaction.getTransactionHash();
            if (ethSendTransaction.hasError()) {
                String message = ethSendTransaction.getError().getMessage();
                System.out.println("transaction failed,info:" + message);
            } else {
                EthGetTransactionReceipt send = web3j.ethGetTransactionReceipt(transactionHash).send();
                if (send != null) {
                    System.out.println("交易成功");
                }
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public static void main(String[] args) {
        SignAndSubmit signAndSubmit = new SignAndSubmit("https://galaxy.block.caduceus.foundation");
        byte[] signMessage = signAndSubmit.offlineSign(
                "私钥",
                "地址",
                new BigInteger("1000000000000000000"));
        if (Objects.isNull(signMessage)){
            System.exit(0);
        }
        signAndSubmit.submitSign(signMessage);
        web3j.shutdown();
    }
}
