package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"net/http"

	"github.com/firebase/genkit/go/ai"
	"github.com/firebase/genkit/go/genkit"
	"github.com/wade-liwei/agentchat/plugins/deepseek"
	"github.com/wade-liwei/agentchat/plugins/graphrag"
	"github.com/wade-liwei/agentchat/plugins/milvus"

	"github.com/firebase/genkit/go/plugins/evaluators"
	"github.com/firebase/genkit/go/plugins/googlegenai"
	"github.com/firebase/genkit/go/plugins/server"

	httpSwagger "github.com/swaggo/http-swagger"
	_ "github.com/wade-liwei/agentchat/docs" // 导入生成的 Swagger 文档
)

//  GraphKnowledge

type Input struct {
	Content  string `json:"content,omitempty"`
	Model    string `json:"model,omitempty"`
	APIKey   string `json:"apiKey,omitempty"`
	Username string `json:"username,omitempty"`
	UserID   string `json:"user_id,omitempty"`
}

// DocumentInput 结构体用于文档索引接口
type DocumentInput struct {
	// UserID   string                 `json:"user_id"`
	// Username string                 `json:"username"`
	Content  string                 `json:"content"`
	Metadata map[string]interface{} `json:"metadata,omitempty"`
}

// GraphInput 结构体用于文档索引接口
type GraphInput struct {
	UserID   string                 `json:"user_id"`
	Username string                 `json:"username"`
	Content  string                 `json:"content"`
	Metadata map[string]interface{} `json:"metadata,omitempty"`
}

func main() {

	ctx := context.Background()

	ds := deepseek.DeepSeek{
		APIKey: "sk-9f70df871a7c4b8aa566a3c7a0603706",
	}

	mil := milvus.Milvus{
		Addr: "54.92.111.204:19530", // Milvus gRPC endpoint
	}
	metrics := []evaluators.MetricConfig{
		{
			MetricType: evaluators.EvaluatorDeepEqual,
		},
		{
			MetricType: evaluators.EvaluatorRegex,
		},
		{
			MetricType: evaluators.EvaluatorJsonata,
		},
	}

	graph := graphrag.GraphKnowledge{
		Addr: "54.92.111.204:5670",
	}

	g, err := genkit.Init(ctx, genkit.WithPlugins(&ds, &mil, &graph, &googlegenai.GoogleAI{APIKey: "AIzaSyCoYBOmnwRWlH_-nT25lpn8pMg3T18Q0uI"}, &evaluators.GenkitEval{Metrics: metrics}))
	if err != nil {
		log.Fatal(err)
	}

	m := ds.DefineModel(g,
		deepseek.ModelDefinition{
			Name: "deepseek-chat", // Choose an appropriate model
			Type: "chat",          // Must be chat for tool support
		},
		nil)

	embedder := googlegenai.GoogleAIEmbedder(g, "embedding-001")
	if embedder == nil {
		log.Fatal("embedder is not defined")
	}

	// Configure collection
	cfg := milvus.CollectionConfig{
		Collection:      "useridx",
		Dimension:       768, // Match mock embedder dimension
		Embedder:        embedder,
		EmbedderOptions: map[string]interface{}{}, // Explicitly set as map
	}

	// Define indexer and retriever
	indexer, retriever, err := milvus.DefineIndexerAndRetriever(ctx, g, cfg)
	if err != nil {
		log.Fatalf("DefineIndexerAndRetriever failed: %v", err)
	}

	_ = retriever

	// 定义文档索引流
	genkit.DefineFlow(g, "indexDocuments", func(ctx context.Context, input *DocumentInput) (string, error) {
		doc := ai.DocumentFromText(input.Content, input.Metadata)
		err := indexer.Index(ctx, &ai.IndexerRequest{
			Documents: []*ai.Document{doc},
		})
		if err != nil {
			return "", fmt.Errorf("index document: %w", err)
		}
		return "Document indexed successfully", nil
	})

	graphIndexer, graphRetriever, err := graphrag.DefineIndexerAndRetriever(ctx, g)

	_ = graphRetriever

	genkit.DefineFlow(g, "indexGraph", func(ctx context.Context, input *GraphInput) (string, error) {

		opt := graphrag.IndexReqOption{
			UserId:   input.UserID,
			UserName: input.Username,
		}

		resDocName := ""

		if v, ok := input.Metadata[graphrag.DocNameKey]; !ok {
			// Generate random docName.
			docName, err := graphrag.GenerateRandomDocName(8)
			if err != nil {
				return "", fmt.Errorf("generate random docName for document  %w", err)
			}
			input.Metadata[graphrag.DocNameKey] = docName
			resDocName = docName
		} else {
			if str, isString := v.(string); isString {
				resDocName = str
			}
		}

		doc := ai.DocumentFromText(input.Content, input.Metadata)
		err := graphIndexer.Index(ctx, &ai.IndexerRequest{
			Documents: []*ai.Document{doc},
			Options:   &opt,
		})
		if err != nil {
			return "", fmt.Errorf("index document: %w", err)
		}
		return fmt.Sprintf("Document indexed successfully, docname %s", resDocName), nil
	})

	// Define a simple flow that generates jokes about a given topic
	genkit.DefineFlow(g, "chat", func(ctx context.Context, input *Input) (string, error) {

		inputAsJson, err := json.Marshal(input)

		if err != nil {
			return "", err
		}

		fmt.Println("input-------------------------------", string(inputAsJson))

		resp, err := genkit.Generate(ctx, g,
			ai.WithModel(m),
			ai.WithPrompt(`Tell silly short jokes about apple`))

		if err != nil {
			fmt.Println(err.Error())
			return "", err
		}

		fmt.Println("resp.Text()", resp.Text())

		if err != nil {
			return "", err
		}

		text := resp.Text()
		return text, nil
	})

	// 配置限速器：每秒 10 次请求，突发容量 20，最大并发 5
	rl := NewRateLimiter(10, 20, 5)

	// 创建 Genkit HTTP 处理器
	mux := http.NewServeMux()

	for _, a := range genkit.ListFlows(g) {
		handler := rl.Middleware(genkit.Handler(a))
		mux.Handle("POST /"+a.Name(), handler)
	}

	// 暴露 Swagger UI，使用 swagger.yaml
	mux.HandleFunc("/swagger/", httpSwagger.Handler(
		httpSwagger.URL("/docs/swagger.yaml"), // 指定 YAML 文件路径
	))

	// 确保 docs 目录可通过 HTTP 访问
	mux.Handle("/docs/", http.StripPrefix("/docs/", http.FileServer(http.Dir("docs"))))

	// 启动服务器，监听
	log.Printf("Server starting on 0.0.0.0:8000")
	if err := server.Start(ctx, "0.0.0.0:8000", mux); err != nil {
		log.Fatalf("Server failed: %v", err)
	}
}
