package knowledge

import (
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"sync"

	"github.com/rs/zerolog/log"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/errors"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
)

// ClientConfig holds configuration options for the Tencent Cloud LKEAP client.
type ClientConfig struct {
	SecretID  string // Tencent Cloud Secret ID
	SecretKey string // Tencent Cloud Secret Key
	Token     string // Optional: Temporary token for authentication
	Endpoint  string // API endpoint (default: lkeap.tencentcloudapi.com)
	Region    string // Tencent Cloud region (optional)
}

// KnowledgeClient manages interactions with the Tencent Cloud LKEAP API.
type KnowledgeClient struct {
	client  *lkeap.Client
	config  ClientConfig
	mu      sync.Mutex
	initted bool
}

// NewKnowledgeClient creates a new KnowledgeClient with the given configuration.
func NewKnowledgeClient(config ClientConfig) *KnowledgeClient {
	log.Info().
		Str("method", "NewKnowledgeClient").
		Str("endpoint", config.Endpoint).
		Str("region", config.Region).
		Str("secret_id", maskCredential(config.SecretID)).
		Str("token", maskCredential(config.Token)).
		Msg("Creating new KnowledgeClient")
	return &KnowledgeClient{
		config: config,
	}
}

// Init initializes the KnowledgeClient.
func (kc *KnowledgeClient) Init(ctx context.Context) error {
	log.Info().Str("method", "KnowledgeClient.Init").Msg("Initializing KnowledgeClient")
	kc.mu.Lock()
	defer kc.mu.Unlock()

	if kc.initted {
		log.Error().Str("method", "KnowledgeClient.Init").Msg("Client already initialized")
		return fmt.Errorf("knowledge client already initialized")
	}

	// Load configuration from environment variables if not set
	if kc.config.SecretID == "" {
		kc.config.SecretID = os.Getenv("TENCENTCLOUD_SECRET_ID")
	}
	if kc.config.SecretKey == "" {
		kc.config.SecretKey = os.Getenv("TENCENTCLOUD_SECRET_KEY")
	}
	if kc.config.Token == "" {
		kc.config.Token = os.Getenv("TENCENTCLOUD_TOKEN")
	}
	if kc.config.Endpoint == "" {
		kc.config.Endpoint = "lkeap.tencentcloudapi.com"
	}
	if kc.config.Region == "" {
		kc.config.Region = "ap-guangzhou" // Default to ap-guangzhou as per curl
	}

	// Validate configuration
	if kc.config.SecretID == "" || kc.config.SecretKey == "" {
		log.Error().Str("method", "KnowledgeClient.Init").Msg("SecretID and SecretKey are required")
		return fmt.Errorf("knowledge: SecretID and SecretKey are required")
	}

	// Create credential
	var credential *common.Credential
	if kc.config.Token != "" {
		credential = common.NewTokenCredential(kc.config.SecretID, kc.config.SecretKey, kc.config.Token)
		log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using temporary token credential")
	} else {
		credential = common.NewCredential(kc.config.SecretID, kc.config.SecretKey)
		log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using standard credential")
	}

	// Create client profile
	cpf := profile.NewClientProfile()
	cpf.HttpProfile.Endpoint = kc.config.Endpoint

	// Initialize client
	client, err := lkeap.NewClient(credential, kc.config.Region, cpf)
	if err != nil {
		log.Error().
			Err(err).
			Str("method", "KnowledgeClient.Init").
			Msg("Failed to create LKEAP client")
		return err
	}

	kc.client = client
	kc.initted = true
	log.Info().Str("method", "KnowledgeClient.Init").Msg("Initialization successful")
	return nil
}

// QueryRewriteRequest defines the input for a query rewrite operation.
type QueryRewriteRequest struct {
	Messages []*lkeap.Message // Multi-turn conversation history (up to 4 turns)
	Model    string           // Model name for query rewriting
}

// QueryRewriteResponse defines the output of a query rewrite operation.
type QueryRewriteResponse struct {
	RewrittenQuery string // The rewritten query
	RawResponse    *lkeap.QueryRewriteResponse
}

// QueryRewrite performs a query rewrite using the Tencent Cloud LKEAP API.
func (kc *KnowledgeClient) QueryRewrite(ctx context.Context, req QueryRewriteRequest) (*QueryRewriteResponse, error) {
	log.Info().
		Str("method", "KnowledgeClient.QueryRewrite").
		Int("message_count", len(req.Messages)).
		Str("model", req.Model).
		Msg("Starting query rewrite operation")

	if !kc.initted {
		log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("Client not initialized")
		return nil, fmt.Errorf("knowledge client not initialized; call Init first")
	}

	// Validate input
	if len(req.Messages) == 0 {
		log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("At least one message is required")
		return nil, fmt.Errorf("at least one message is required")
	}
	if len(req.Messages) > 4 {
		log.Warn().
			Str("method", "KnowledgeClient.QueryRewrite").
			Int("message_count", len(req.Messages)).
			Msg("Message count exceeds 4, truncating to 4")
		req.Messages = req.Messages[:4]
	}
	for i, msg := range req.Messages {
		if msg.Role == nil || *msg.Role == "" {
			log.Error().
				Str("method", "KnowledgeClient.QueryRewrite").
				Int("index", i).
				Msg("Role is required in each message")
			return nil, fmt.Errorf("message at index %d missing role", i)
		}
		if *msg.Role != "user" && *msg.Role != "assistant" {
			log.Error().
				Str("method", "KnowledgeClient.QueryRewrite").
				Int("index", i).
				Str("role", *msg.Role).
				Msg("Invalid role; must be 'user' or 'assistant'")
			return nil, fmt.Errorf("invalid role '%s' at index %d", *msg.Role, i)
		}
		if msg.Content == nil || *msg.Content == "" {
			log.Error().
				Str("method", "KnowledgeClient.QueryRewrite").
				Int("index", i).
				Msg("Content is required in each message")
			return nil, fmt.Errorf("message at index %d missing content", i)
		}
		log.Debug().
			Str("method", "KnowledgeClient.QueryRewrite").
			Int("index", i).
			Str("role", *msg.Role).
			Str("content", *msg.Content).
			Msg("Validated message")
	}
	if req.Model == "" {
		log.Warn().Str("method", "KnowledgeClient.QueryRewrite").Msg("Model not specified, using default")
		req.Model = "lke-query-rewrite-base" // Default as per curl
	}

	// Create Tencent Cloud request
	tencentReq := lkeap.NewQueryRewriteRequest()
	tencentReq.Messages = req.Messages
	if req.Model != "" {
		tencentReq.Model = common.StringPtr(req.Model)
	}

	// Debug request
	tencentReqAsJson, _ := json.Marshal(tencentReq)
	log.Debug().
		Str("method", "KnowledgeClient.QueryRewrite").
		Str("request_json", string(tencentReqAsJson)).
		Msg("Prepared Tencent Cloud request")

	// Perform request
	response, err := kc.client.QueryRewriteWithContext(ctx, tencentReq)
	if err != nil {
		if _, ok := err.(*errors.TencentCloudSDKError); ok {
			log.Error().
				Err(err).
				Str("method", "KnowledgeClient.QueryRewrite").
				Msg("Tencent Cloud API error")
			return nil, fmt.Errorf("tencent cloud api error: %w", err)
		}
		log.Error().
			Err(err).
			Str("method", "KnowledgeClient.QueryRewrite").
			Msg("Failed to perform query rewrite")
		return nil, fmt.Errorf("query rewrite failed: %w", err)
	}

	// Extract response fields
	var rewrittenQuery string
	var requestId string
	if response.Response.Content != nil {
		rewrittenQuery = *response.Response.Content
	}
	if response.Response.RequestId != nil {
		requestId = *response.Response.RequestId
	}

	result := &QueryRewriteResponse{
		RewrittenQuery: rewrittenQuery,
		RawResponse:    response,
	}

	log.Info().
		Str("method", "KnowledgeClient.QueryRewrite").
		Str("rewritten_query", rewrittenQuery).
		Str("request_id", requestId).
		Interface("usage", response.Response.Usage).
		Str("raw_response", response.ToJsonString()).
		Msg("Query rewrite operation completed successfully")
	return result, nil
}

// maskCredential masks sensitive credentials for logging
func maskCredential(cred string) string {
	if len(cred) <= 8 {
		return strings.Repeat("*", len(cred))
	}
	return cred[:4] + strings.Repeat("*", len(cred)-8) + cred[len(cred)-4:]
}

// QueryRewriteWithSummary wraps QueryRewrite to handle a user question, assistant answer, and history summary.
func (kc *KnowledgeClient) QueryRewriteWithSummary(ctx context.Context, userQuestion, assistantAnswer, historySummary string) (*QueryRewriteResponse, error) {
	log.Info().
		Str("method", "KnowledgeClient.QueryRewriteWithSummary").
		Str("user_question", userQuestion).
		Str("assistant_answer", assistantAnswer).
		Str("history_summary", historySummary).
		Msg("Starting query rewrite with summary operation")

	if userQuestion == "" || assistantAnswer == "" {
		log.Error().Str("method", "KnowledgeClient.QueryRewriteWithSummary").Msg("User question and assistant answer are required")
		return nil, fmt.Errorf("user question and assistant answer are required")
	}

	// Construct messages
	messages := []*lkeap.Message{
		{
			Role:    common.StringPtr("user"),
			Content: common.StringPtr(userQuestion),
		},
		{
			Role:    common.StringPtr("assistant"),
			Content: common.StringPtr(assistantAnswer),
		},
	}

	// Append history summary as an assistant message if provided
	if historySummary != "" {
		messages = append(messages, &lkeap.Message{
			Role:    common.StringPtr("user"),
			Content: common.StringPtr(fmt.Sprintf("Conversation summary: %s", historySummary)),
		})
	}

	// Create request
	req := QueryRewriteRequest{
		Messages: messages,
		Model:    "lke-query-rewrite-base",
	}

	// Call QueryRewrite
	return kc.QueryRewrite(ctx, req)
}

















// ReconstructDocumentSSE performs document reconstruction using the Tencent Cloud LKEAP API.
// It takes a text string, encodes it as Base64, and returns the reconstructed content as a slice of ReconstructDocumentSSEResponse structs for events with Progress "100".
func (kc *KnowledgeClient) ReconstructDocumentSSE(ctx context.Context, text string) ([]ReconstructDocumentSSEResponse, error) {
    log.Info().
        Str("method", "KnowledgeClient.ReconstructDocumentSSE").
        Str("text_prefix", text[:min(32, len(text))]).
        Msg("Starting document reconstruction operation")

    if !kc.initted {
        log.Error().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Client not initialized")
        return nil, fmt.Errorf("knowledge client not initialized; call Init first")
    }

    // Validate input
    if text == "" {
        log.Error().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Text is required")
        return nil, fmt.Errorf("text is required")
    }

    // Encode text to Base64
    base64Data := base64.StdEncoding.EncodeToString([]byte(text))
    // Check Base64 size (8MB limit)
    if len(base64Data)/4*3 > 8*1024*1024 {
        log.Error().Str("method", "KnowledgeClient.ReconstructDocumentSSE").Msg("Encoded text exceeds 8MB limit")
        return nil, fmt.Errorf("encoded text exceeds 8MB limit")
    }

    // Remove Base64 prefix if present
    if strings.HasPrefix(base64Data, "data:application/octet-stream;base64,") {
        base64Data = strings.TrimPrefix(base64Data, "data:application/octet-stream;base64,")
        log.Debug().
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Msg("Removed Base64 data URI prefix")
    }

    // Create SDK request
    request := lkeap.NewReconstructDocumentSSERequest()
    request.FileType = common.StringPtr("TXT")
    request.FileBase64 = common.StringPtr(base64Data)

    // Debug request
    requestAsJson, _ := json.Marshal(request)
    log.Debug().
        Str("method", "KnowledgeClient.ReconstructDocumentSSE").
        Str("request_json", string(requestAsJson)).
        Msg("Prepared Tencent Cloud request")

    // Perform request
    response, err := kc.client.ReconstructDocumentSSEWithContext(ctx, request)
    if _, ok := err.(*errors.TencentCloudSDKError); ok {
        log.Error().
            Err(err).
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Msg("Tencent Cloud API error")
        return nil, fmt.Errorf("tencent cloud api error: %s", err)
    }
    if err != nil {
        log.Error().
            Err(err).
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Msg("Failed to perform document reconstruction")
        return nil, fmt.Errorf("document reconstruction failed: %w", err)
    }

    // Handle streaming response
    var results []ReconstructDocumentSSEResponse
    for event := range response.Events {
        var recon ReconstructDocumentSSEResponse
        eventData := event.Data
        if len(eventData) == 0 {
            log.Warn().
                Str("method", "KnowledgeClient.ReconstructDocumentSSE").
                Msg("Empty event data in streaming response")
            continue
        }

        if err := json.Unmarshal(eventData, &recon); err != nil {
            log.Error().
                Err(err).
                Str("method", "KnowledgeClient.ReconstructDocumentSSE").
                Str("event_data", string(eventData)).
                Msg("Failed to parse streaming event")
            continue
        }

        // Only collect events with Progress == "100"
        if recon.Progress != nil && *recon.Progress == "100" {
            results = append(results, recon)
            log.Debug().
                Str("method", "KnowledgeClient.ReconstructDocumentSSE").
                Str("event_data", string(eventData)).
                Msg("Collected streaming event with Progress 100")
        } else {
            log.Debug().
                Str("method", "KnowledgeClient.ReconstructDocumentSSE").
                Str("event_data", string(eventData)).
                Msg("Skipped streaming event with Progress != 100")
        }
    }

    // Check if any valid results were collected
    if len(results) == 0 {
        log.Error().
            Str("method", "KnowledgeClient.ReconstructDocumentSSE").
            Msg("No events with Progress 100 returned")
        return nil, fmt.Errorf("no events with progress 100 returned")
    }

    log.Info().
        Str("method", "KnowledgeClient.ReconstructDocumentSSE").
        Int("results_count", len(results)).
        Msg("Document reconstruction operation completed successfully")

    return results, nil
}

// min returns the minimum of two integers
func min(a, b int) int {
    if a < b {
        return a
    }
    return b
}



// ReconstructDocumentSSEResponse defines the response structure for the ReconstructDocumentSSE API.
type ReconstructDocumentSSEResponse struct {
    RequestId                  *string   `json:"RequestId,omitempty"`
    TaskId                     *string   `json:"TaskId,omitempty"`
    ResponseType               *string   `json:"ResponseType,omitempty"`
    Progress                   *string   `json:"Progress,omitempty"`
    ProgressMessage            *string   `json:"ProgressMessage,omitempty"`
    DocumentRecognizeResultUrl *string   `json:"DocumentRecognizeResultUrl,omitempty"`
    FailedPages                []*int64  `json:"FailedPages,omitempty"`
    StatusCode                 *string   `json:"StatusCode,omitempty"`
}

