// Copyright 2025
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

package knowledge

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"sync"

	"github.com/rs/zerolog/log"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/errors"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
)

// ClientConfig holds configuration options for the Tencent Cloud LKEAP client.
type ClientConfig struct {
    SecretID  string // Tencent Cloud Secret ID
    SecretKey string // Tencent Cloud Secret Key
    Token     string // Optional: Temporary token for authentication
    Endpoint  string // API endpoint (default: lkeap.tencentcloudapi.com)
    Region    string // Tencent Cloud region (optional)
}

// KnowledgeClient manages interactions with the Tencent Cloud LKEAP API.
type KnowledgeClient struct {
    client  *lkeap.Client
    config  ClientConfig
    mu      sync.Mutex
    initted bool
}

// NewKnowledgeClient creates a new KnowledgeClient with the given configuration.
func NewKnowledgeClient(config ClientConfig) *KnowledgeClient {
    log.Info().
        Str("method", "NewKnowledgeClient").
        Str("endpoint", config.Endpoint).
        Str("region", config.Region).
        Msg("Creating new KnowledgeClient")
    return &KnowledgeClient{
        config: config,
    }
}

// Init initializes the KnowledgeClient.
func (kc *KnowledgeClient) Init(ctx context.Context) error {
    log.Info().Str("method", "KnowledgeClient.Init").Msg("Initializing KnowledgeClient")
    kc.mu.Lock()
    defer kc.mu.Unlock()

    if kc.initted {
        log.Error().Str("method", "KnowledgeClient.Init").Msg("Client already initialized")
        return fmt.Errorf("knowledge client already initialized")
    }

    // Load configuration defaults
    if kc.config.SecretID == "" {
        kc.config.SecretID = os.Getenv("TENCENTCLOUD_SECRET_ID")
    }
    if kc.config.SecretKey == "" {
        kc.config.SecretKey = os.Getenv("TENCENTCLOUD_SECRET_KEY")
    }
    if kc.config.Token == "" {
        kc.config.Token = os.Getenv("TENCENTCLOUD_TOKEN")
    }
    if kc.config.Endpoint == "" {
        kc.config.Endpoint = "lkeap.tencentcloudapi.com"
    }
    if kc.config.Region == "" {
        kc.config.Region = "" // Region can be empty for global APIs
    }

    // Validate configuration
    if kc.config.SecretID == "" || kc.config.SecretKey == "" {
        log.Error().Str("method", "KnowledgeClient.Init").Msg("SecretID and SecretKey are required")
        return fmt.Errorf("knowledge: SecretID and SecretKey are required")
    }

    // Create credential
    var credential *common.Credential
    if kc.config.Token != "" {
        credential = common.NewTokenCredential(kc.config.SecretID, kc.config.SecretKey, kc.config.Token)
        log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using temporary token credential")
    } else {
        credential = common.NewCredential(kc.config.SecretID, kc.config.SecretKey)
        log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using standard credential")
    }

    // Create client profile
    cpf := profile.NewClientProfile()
    cpf.HttpProfile.Endpoint = kc.config.Endpoint

    // Initialize client
    client, err := lkeap.NewClient(credential, kc.config.Region, cpf)
    if err != nil {
        log.Error().
            Err(err).
            Str("method", "KnowledgeClient.Init").
            Msg("Failed to create LKEAP client")
        return err
    }

    kc.client = client
    kc.initted = true
    log.Info().Str("method", "KnowledgeClient.Init").Msg("Initialization successful")
    return nil
}

// QueryRewriteRequest defines the input for a query rewrite operation.
type QueryRewriteRequest struct {
    Messages []*lkeap.Message // Multi-turn conversation history (up to 4 turns)
    Model    string           // Model name for query rewriting
}

// QueryRewriteResponse defines the output of a query rewrite operation.
type QueryRewriteResponse struct {
    RewrittenQuery string // The rewritten query
    RawResponse    *lkeap.QueryRewriteResponse
}

// QueryRewrite performs a query rewrite using the Tencent Cloud LKEAP API.
func (kc *KnowledgeClient) QueryRewrite(ctx context.Context, req QueryRewriteRequest) (*QueryRewriteResponse, error) {
    log.Info().
        Str("method", "KnowledgeClient.QueryRewrite").
        Int("message_count", len(req.Messages)).
        Str("model", req.Model).
        Msg("Starting query rewrite operation")

    if !kc.initted {
        log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("Client not initialized")
        return nil, fmt.Errorf("knowledge client not initialized; call Init first")
    }

    // Validate input
    if len(req.Messages) == 0 {
        log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("At least one message is required")
        return nil, fmt.Errorf("at least one message is required")
    }
    if len(req.Messages) > 4 {
        log.Warn().
            Str("method", "KnowledgeClient.QueryRewrite").
            Int("message_count", len(req.Messages)).
            Msg("Message count exceeds 4, truncating to 4")
        req.Messages = req.Messages[:4]
    }
    for i, msg := range req.Messages {
        if msg.Role == nil || *msg.Role == "" {
            log.Error().
                Str("method", "KnowledgeClient.QueryRewrite").
                Int("index", i).
                Msg("Role is required in each message")
            return nil, fmt.Errorf("message at index %d missing role", i)
        }
        if *msg.Role != "user" && *msg.Role != "assistant" {
            log.Error().
                Str("method", "KnowledgeClient.QueryRewrite").
                Int("index", i).
                Str("role", *msg.Role).
                Msg("Invalid role; must be 'user' or 'assistant'")
            return nil, fmt.Errorf("invalid role '%s' at index %d", *msg.Role, i)
        }
        if msg.Content == nil || *msg.Content == "" {
            log.Error().
                Str("method", "KnowledgeClient.QueryRewrite").
                Int("index", i).
                Msg("Content is required in each message")
            return nil, fmt.Errorf("message at index %d missing content", i)
        }
        log.Debug().
            Str("method", "KnowledgeClient.QueryRewrite").
            Int("index", i).
            Str("role", *msg.Role).
            Str("content", *msg.Content).
            Msg("Validated message")
    }
    if req.Model == "" {
        log.Warn().Str("method", "KnowledgeClient.QueryRewrite").Msg("Model not specified, using default")
    }

    // Create Tencent Cloud request
    tencentReq := lkeap.NewQueryRewriteRequest()
    tencentReq.Messages = req.Messages // Directly use validated messages
    if req.Model != "" {
        tencentReq.Model = common.StringPtr(req.Model)
    }



	fmt.Println("len(tencentReq.Messages)",len(tencentReq.Messages))


	tencentReqAsJson, _ :=  json.Marshal(tencentReq)

	fmt.Println("len(tencentReq.Messages) json marsh",string(tencentReqAsJson))

    // Perform request
    response, err := kc.client.QueryRewriteWithContext(ctx, tencentReq)
    if err != nil {
        if _, ok := err.(*errors.TencentCloudSDKError); ok {
            log.Error().
                Err(err).
                Str("method", "KnowledgeClient.QueryRewrite").
                Msg("Tencent Cloud API error")
            return nil, fmt.Errorf("tencent cloud api error: %w", err)
        }
        log.Error().
            Err(err).
            Str("method", "KnowledgeClient.QueryRewrite").
            Msg("Failed to perform query rewrite")
        return nil, fmt.Errorf("query rewrite failed: %w", err)
    }
// Extract response fields
var rewrittenQuery string
var requestId string
if response.Response.Content != nil {
	rewrittenQuery = *response.Response.Content
}
if response.Response.RequestId != nil {
	requestId = *response.Response.RequestId
}

result := &QueryRewriteResponse{
	RewrittenQuery: rewrittenQuery,
	// Usage:          response.Response.Usage,
	// RequestId:      requestId,
	RawResponse:    response,
}

log.Info().
	Str("method", "KnowledgeClient.QueryRewrite").
	Str("rewritten_query", rewrittenQuery).
	Str("request_id", requestId).
	Interface("usage", response.Response.Usage).
	Str("raw_response", response.ToJsonString()).
	Msg("Query rewrite operation completed successfully")
return result, nil
}


//     // Extract rewritten query
//     var rewrittenQuery string
//     if response.Response.RewrittenQuery != nil {
//         rewrittenQuery = *response.Response.RewrittenQuery
//     }

//     result := &QueryRewriteResponse{
//         RewrittenQuery: rewrittenQuery,
//         RawResponse:    response,
//     }

//     log.Info().
//         Str("method", "KnowledgeClient.QueryRewrite").
//         Str("rewritten_query", rewrittenQuery).
//         Str("raw_response", response.ToJsonString()).
//         Msg("Query rewrite operation completed successfully")
//     return result, nil
// }




















// // Copyright 2025
// //
// // Licensed under the Apache License, Version 2.0 (the "License");
// // you may not use this file except in compliance with the License.
// // You may obtain a copy of the License at
// //
// //     http://www.apache.org/licenses/LICENSE-2.0
// //
// // Unless required by applicable law or agreed to in writing, software
// // distributed under the License is distributed on an "AS IS" BASIS,
// // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// // See the License for the specific language governing permissions and
// // limitations under the License.
// //
// // SPDX-License-Identifier: Apache-2.0

// package knowledge

// import (
// 	"context"
// 	"fmt"
// 	"os"
// 	"sync"

// 	"github.com/rs/zerolog/log"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/errors"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
// 	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
// )

// // ClientConfig holds configuration options for the Tencent Cloud LKEAP client.
// type ClientConfig struct {
//     SecretID  string // Tencent Cloud Secret ID
//     SecretKey string // Tencent Cloud Secret Key
//     Token     string // Optional: Temporary token for authentication
//     Endpoint  string // API endpoint (default: lkeap.tencentcloudapi.com)
//     Region    string // Tencent Cloud region (optional)
// }

// // KnowledgeClient manages interactions with the Tencent Cloud LKEAP API.
// type KnowledgeClient struct {
//     client  *lkeap.Client
//     config  ClientConfig
//     mu      sync.Mutex
//     initted bool
// }

// // NewKnowledgeClient creates a new KnowledgeClient with the given configuration.
// func NewKnowledgeClient(config ClientConfig) *KnowledgeClient {
//     log.Info().
//         Str("method", "NewKnowledgeClient").
//         Str("endpoint", config.Endpoint).
//         Str("region", config.Region).
//         Msg("Creating new KnowledgeClient")
//     return &KnowledgeClient{
//         config: config,
//     }
// }

// // Init initializes the KnowledgeClient.
// func (kc *KnowledgeClient) Init(ctx context.Context) error {
//     log.Info().Str("method", "KnowledgeClient.Init").Msg("Initializing KnowledgeClient")
//     kc.mu.Lock()
//     defer kc.mu.Unlock()

//     if kc.initted {
//         log.Error().Str("method", "KnowledgeClient.Init").Msg("Client already initialized")
//         return errors.New("knowledge client already initialized")
//     }

//     // Load configuration defaults
//     if kc.config.SecretID == "" {
//         kc.config.SecretID = os.Getenv("TENCENTCLOUD_SECRET_ID")
//     }
//     if kc.config.SecretKey == "" {
//         kc.config.SecretKey = os.Getenv("TENCENTCLOUD_SECRET_KEY")
//     }
//     if kc.config.Token == "" {
//         kc.config.Token = os.Getenv("TENCENTCLOUD_TOKEN")
//     }
//     if kc.config.Endpoint == "" {
//         kc.config.Endpoint = "lkeap.tencentcloudapi.com"
//     }
//     if kc.config.Region == "" {
//         kc.config.Region = "" // Region can be empty for global APIs
//     }

//     // Validate configuration
//     if kc.config.SecretID == "" || kc.config.SecretKey == "" {
//         log.Error().Str("method", "KnowledgeClient.Init").Msg("SecretID and SecretKey are required")
//         return fmt.Errorf("knowledge: SecretID and SecretKey are required")
//     }

//     // Create credential
//     var credential *common.Credential
//     if kc.config.Token != "" {
//         credential = common.NewTokenCredential(kc.config.SecretID, kc.config.SecretKey, kc.config.Token)
//         log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using temporary token credential")
//     } else {
//         credential = common.NewCredential(kc.config.SecretID, kc.config.SecretKey)
//         log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using standard credential")
//     }

//     // Create client profile
//     cpf := profile.NewClientProfile()
//     cpf.HttpProfile.Endpoint = kc.config.Endpoint

//     // Initialize client
//     client, err := lkeap.NewClient(credential, kc.config.Region, cpf)
//     if err != nil {
//         log.Error().
//             Err(err).
//             Str("method", "KnowledgeClient.Init").
//             Msg("Failed to create LKEAP client")
//         return err
//     }

//     kc.client = client
//     kc.initted = true
//     log.Info().Str("method", "KnowledgeClient.Init").Msg("Initialization successful")
//     return nil
// }

// // Message represents a single turn in a conversation.
// type Message struct {
//     User      string // User's question
//     Assistant string // Assistant's response
// }

// // QueryRewriteRequest defines the input for a query rewrite operation.
// type QueryRewriteRequest struct {
//     Messages []*Message // Multi-turn conversation history (up to 4 turns)
//     Model    string     // Model name for query rewriting
// }

// // QueryRewriteResponse defines the output of a query rewrite operation.
// type QueryRewriteResponse struct {
//     RewrittenQuery string // The rewritten query
//     RawResponse    *lkeap.QueryRewriteResponse
// }

// // QueryRewrite performs a query rewrite using the Tencent Cloud LKEAP API.
// func (kc *KnowledgeClient) QueryRewrite(ctx context.Context, req QueryRewriteRequest) (*QueryRewriteResponse, error) {
//     log.Info().
//         Str("method", "KnowledgeClient.QueryRewrite").
//         Int("message_count", len(req.Messages)).
//         Str("model", req.Model).
//         Msg("Starting query rewrite operation")

//     if !kc.initted {
//         log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("Client not initialized")
//         return nil, fmt.Errorf("knowledge client not initialized; call Init first")
//     }

//     // Validate input
//     if len(req.Messages) == 0 {
//         log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("At least one message is required")
//         return nil, fmt.Errorf("at least one message is required")
//     }
//     if len(req.Messages) > 4 {
//         log.Warn().
//             Str("method", "KnowledgeClient.QueryRewrite").
//             Int("message_count", len(req.Messages)).
//             Msg("Message count exceeds 4, truncating to 4")
//         req.Messages = req.Messages[:4]
//     }
//     for i, msg := range req.Messages {
//         if msg.User == "" || msg.Assistant == "" {
//             log.Error().
//                 Str("method", "KnowledgeClient.QueryRewrite").
//                 Int("index", i).
//                 Msg("User and Assistant fields are required in each message")
//             return nil, fmt.Errorf("message at index %d missing user or assistant", i)
//         }
//     }
//     if req.Model == "" {
//         log.Warn().Str("method", "KnowledgeClient.QueryRewrite").Msg("Model not specified, using default")
//     }

//     // Create Tencent Cloud request
//     tencentReq := lkeap.NewQueryRewriteRequest()
//     tencentReq.Messages = make([]*lkeap.Message, len(req.Messages))
//     for i, msg := range req.Messages {
//         tencentReq.Messages[i] = &lkeap.Message{
//             User:      common.StringPtr(msg.User),
//             Assistant: common.StringPtr(msg.Assistant),
//         }
//         log.Debug().
//             Str("method", "KnowledgeClient.QueryRewrite").
//             Int("index", i).
//             Str("user", msg.User).
//             Str("assistant", msg.Assistant).
//             Msg("Added message to request")
//     }
//     if req.Model != "" {
//         tencentReq.Model = common.StringPtr(req.Model)
//     }

//     // Perform request
//     response, err := kc.client.QueryRewriteWithContext(ctx, tencentReq)
//     if err != nil {
//         if _, ok := err.(*errors.TencentCloudSDKError); ok {
//             log.Error().
//                 Err(err).
//                 Str("method", "KnowledgeClient.QueryRewrite").
//                 Msg("Tencent Cloud API error")
//             return nil, fmt.Errorf("tencent cloud api error: %w", err)
//         }
//         log.Error().
//             Err(err).
//             Str("method", "KnowledgeClient.QueryRewrite").
//             Msg("Failed to perform query rewrite")
//         return nil, fmt.Errorf("query rewrite failed: %w", err)
//     }

//     // Extract rewritten query
//     var rewrittenQuery string
//     if response.Response.RewrittenQuery != nil {
//         rewrittenQuery = *response.Response.RewrittenQuery
//     }

//     result := &QueryRewriteResponse{
//         RewrittenQuery: rewrittenQuery,
//         RawResponse:    response,
//     }

//     log.Info().
//         Str("method", "KnowledgeClient.QueryRewrite").
//         Str("rewritten_query", rewrittenQuery).
//         Str("raw_response", response.ToJsonString()).
//         Msg("Query rewrite operation completed successfully")
//     return result, nil
// }
























// // Copyright 2025
// //
// // Licensed under the Apache License, Version 2.0 (the "License");
// // you may not use this file except in compliance with the License.
// // You may obtain a copy of the License at
// //
// //     http://www.apache.org/licenses/LICENSE-2.0
// //
// // Unless required by applicable law or agreed to in writing, software
// // distributed under the License is distributed on an "AS IS" BASIS,
// // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// // See the License for the specific language governing permissions and
// // limitations under the License.
// //
// // SPDX-License-Identifier: Apache-2.0

// package knowledge

// import (
// 	"context"
// 	"fmt"
// 	"os"
// 	"sync"

// 	"github.com/rs/zerolog/log"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/errors"
// 	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
// 	lkeap "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/lkeap/v20240522"
// )

// // ClientConfig holds configuration options for the Tencent Cloud LKEAP client.
// type ClientConfig struct {
//     SecretID  string // Tencent Cloud Secret ID
//     SecretKey string // Tencent Cloud Secret Key
//     Token     string // Optional: Temporary token for authentication
//     Endpoint  string // API endpoint (default: lkeap.tencentcloudapi.com)
//     Region    string // Tencent Cloud region (optional)
// }

// // KnowledgeClient manages interactions with the Tencent Cloud LKEAP API.
// type KnowledgeClient struct {
//     client  *lkeap.Client
//     config  ClientConfig
//     mu      sync.Mutex
//     initted bool
// }

// // NewKnowledgeClient creates a new KnowledgeClient with the given configuration.
// func NewKnowledgeClient(config ClientConfig) *KnowledgeClient {
//     log.Info().
//         Str("method", "NewKnowledgeClient").
//         Str("endpoint", config.Endpoint).
//         Str("region", config.Region).
//         Msg("Creating new KnowledgeClient")
//     return &KnowledgeClient{
//         config: config,
//     }
// }

// // Init initializes the KnowledgeClient.
// func (kc *KnowledgeClient) Init(ctx context.Context) error {
//     log.Info().Str("method", "KnowledgeClient.Init").Msg("Initializing KnowledgeClient")
//     kc.mu.Lock()
//     defer kc.mu.Unlock()

//     if kc.initted {
//         log.Error().Str("method", "KnowledgeClient.Init").Msg("Client already initialized")
// 		return fmt.Errorf("knowledge client already initialized")
//     }

//     // Load configuration defaults
//     if kc.config.SecretID == "" {
//         kc.config.SecretID = os.Getenv("TENCENTCLOUD_SECRET_ID")
//     }
//     if kc.config.SecretKey == "" {
//         kc.config.SecretKey = os.Getenv("TENCENTCLOUD_SECRET_KEY")
//     }
//     if kc.config.Token == "" {
//         kc.config.Token = os.Getenv("TENCENTCLOUD_TOKEN")
//     }
//     if kc.config.Endpoint == "" {
//         kc.config.Endpoint = "lkeap.tencentcloudapi.com"
//     }
//     if kc.config.Region == "" {
//         kc.config.Region = "" // Region can be empty for global APIs
//     }

//     // Validate configuration
//     if kc.config.SecretID == "" || kc.config.SecretKey == "" {
//         log.Error().Str("method", "KnowledgeClient.Init").Msg("SecretID and SecretKey are required")
//         //return errors.New("knowledge: SecretID and SecretKey are required")
// 		return fmt.Errorf("knowledge: SecretID and SecretKey are required")
// 	}

//     // Create credential
//     var credential *common.Credential
//     if kc.config.Token != "" {
//         credential = common.NewTokenCredential(kc.config.SecretID, kc.config.SecretKey, kc.config.Token)
//         log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using temporary token credential")
//     } else {
//         credential = common.NewCredential(kc.config.SecretID, kc.config.SecretKey)
//         log.Debug().Str("method", "KnowledgeClient.Init").Msg("Using standard credential")
//     }

//     // Create client profile
//     cpf := profile.NewClientProfile()
//     cpf.HttpProfile.Endpoint = kc.config.Endpoint

//     // Initialize client
//     client, err := lkeap.NewClient(credential, kc.config.Region, cpf)
//     if err != nil {
//         log.Error().
//             Err(err).
//             Str("method", "KnowledgeClient.Init").
//             Msg("Failed to create LKEAP client")
//         return err
//     }

//     kc.client = client
//     kc.initted = true
//     log.Info().Str("method", "KnowledgeClient.Init").Msg("Initialization successful")
//     return nil
// }

// // QueryRewriteRequest defines the input for a query rewrite operation.
// type QueryRewriteRequest struct {
//     Query string // The input query to rewrite
// }

// // QueryRewriteResponse defines the output of a query rewrite operation.
// type QueryRewriteResponse struct {
//     RewrittenQuery string // The rewritten query
//     RawResponse    *lkeap.QueryRewriteResponse
// }

// // QueryRewrite performs a query rewrite using the Tencent Cloud LKEAP API.
// func (kc *KnowledgeClient) QueryRewrite(ctx context.Context, req QueryRewriteRequest) (*QueryRewriteResponse, error) {
//     log.Info().
//         Str("method", "KnowledgeClient.QueryRewrite").
//         Str("query", req.Query).
//         Msg("Starting query rewrite operation")

//     if !kc.initted {
//         log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("Client not initialized")
//         //return nil, errors.New("knowledge client not initialized; call Init first")

// 		return nil,fmt.Errorf("knowledge client not initialized; call Init first")
//     }

//     if req.Query == "" {
//         log.Error().Str("method", "KnowledgeClient.QueryRewrite").Msg("Query is required")
//         //return nil, errors.New("query is required")
// 		return nil, fmt.Errorf("query is required")
// 	}

//     // Create request
//     tencentReq := lkeap.NewQueryRewriteRequest()
//     tencentReq.Query = common.StringPtr(req.Query)

//     // Perform request
//     response, err := kc.client.QueryRewriteWithContext(ctx, tencentReq)
//     if err != nil {
//         if _, ok := err.(*errors.TencentCloudSDKError); ok {
//             log.Error().
//                 Err(err).
//                 Str("method", "KnowledgeClient.QueryRewrite").
//                 Msg("Tencent Cloud API error")
//             return nil, fmt.Errorf("tencent cloud api error: %w", err)
//         }
//         log.Error().
//             Err(err).
//             Str("method", "KnowledgeClient.QueryRewrite").
//             Msg("Failed to perform query rewrite")
//         return nil, fmt.Errorf("query rewrite failed: %w", err)
//     }

//     // Extract rewritten query
//     var rewrittenQuery string
//     if response.Response.RewrittenQuery != nil {
//         rewrittenQuery = *response.Response.RewrittenQuery
//     }

//     result := &QueryRewriteResponse{
//         RewrittenQuery: rewrittenQuery,
//         RawResponse:    response,
//     }

//     log.Info().
//         Str("method", "KnowledgeClient.QueryRewrite").
//         Str("rewritten_query", rewrittenQuery).
//         Str("raw_response", response.ToJsonString()).
//         Msg("Query rewrite operation completed successfully")
//     return result, nil
// }




